package visrd;


/**

   This file is part of VisRD

   Copyright (C) 2002 Kristoffer Forslund (jeanpaulsartre@hotmail.com)
   for the Linnaeus Centre for Bioinformatics, Uppsala University

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with VisRD; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/**

	Class with static functions for generating a starting set of quartets

*/

public class WeightedStatisticalGeometryCalculator extends TrajectoryCalculator {
	private DistanceFunction d; // Distance function

	// Constructor - loads with the characters
	public WeightedStatisticalGeometryCalculator (Taxon[] taxon, DistanceFunction newD){
		super(taxon);
		d = newD;
	}
	
	// Heart of the algorithm
	public Triplet processWindow(int s, char[] rowX, char[] rowY, char[] rowU, char[] rowV, Quartet q){
		double c1 = 0;
		double c2 = 0;
		double c3 = 0;
	
		for (int wS = s - windowSize / 2; wS < s + windowSize / 2; wS++) {
			if ((! useMask) || (! mask[wS])) { // If position is not masked...
			
				// ... then, check whether the things equal or not...
			
				char x = Taxon.resolveCharacter(rowX[wS]);
				char y = Taxon.resolveCharacter(rowY[wS]);
				char u = Taxon.resolveCharacter(rowU[wS]);
				char v = Taxon.resolveCharacter(rowV[wS]);

				// Three sums
				// All possible ways of picking two elements from four.
				double s1 = d.delta (x, y) + d.delta (u, v);
				double s2 = d.delta (x, u) + d.delta (y, v);
				double s3 = d.delta (x, v) + d.delta (u, y);		
							
				double L = Math.max (s1, Math.max (s2, s3));
							
				c1 += (L - s1) / 2;
				c2 += (L - s2) / 2;
				c3 += (L - s3) / 2;			
			}
		}

		// Now we have the counts, create the weights and the triplet
		return generateTriplet(s,c1,c2,c3);
	}
}

class DistanceCalculator extends TrajectoryCalculator {
	private int type;
	public static final int JC=2, K2P=3, F84=4, TN=5, PROTEIN=6; // Number corresponds to position in scan method combo box
	
	private double gFrequency, cFrequency, aFrequency, tFrequency;

	private static final boolean useWholeAlignmentForFrequencies=false;
	private double alpha = 0.5; // for gamma rates
	private static final boolean USE_LEAST_SQUARED=false;
		
/*	
	public DistanceCalculator (Taxon[] taxon, int type){
		super(taxon);
		this.type=type;
		
		System.out.println("type=[" + type + "]");
	}
*/	
	public DistanceCalculator (Taxon[] taxon, int type){
		super(taxon);
		this.type=type;

		windowSize = VisRD.getWindowSize();
		if(useWholeAlignmentForFrequencies){
			int aCount=0,cCount=0,gCount=0,tCount=0;
			
			for(int t=0;t<taxon.length;t++){
				char[] sequence = taxon[t].getMatrix();
				for (int wS = 0;wS<sequence.length;wS++) {
					if ((! useMask) || (! mask[wS])) { // If position is not masked...
						switch(sequence[wS]){
							case 'g': gCount++; break;
							case 'c': cCount++; break;
							case 'a': aCount++; break;
							case 't': tCount++; break;
						}
					}
				}
			}
			
			int totalChars = taxon[0].getNchar()*taxon.length;
	
			// Looking accross 4 taxa i.e. X,Y,U,V
			gFrequency = (double)gCount/((double)totalChars);
			cFrequency = (double)cCount/((double)totalChars);
			aFrequency = (double)aCount/((double)totalChars);
			tFrequency = (double)tCount/((double)totalChars);
		}
	}
	
	// Heart of the algorithm
	public Triplet processWindow(int s, char[] rowX, char[] rowY, char[] rowU, char[] rowV, Quartet q){
		if(!useWholeAlignmentForFrequencies){
			int aCount=0,cCount=0,gCount=0,tCount=0;
			
			for (int wS = s - windowSize / 2; wS < s + windowSize / 2; wS++) {
				if ((! useMask) || (! mask[wS])) { // If position is not masked...
					char[] chars= {Taxon.resolveCharacter(rowX[wS]),Taxon.resolveCharacter(rowY[wS]),Taxon.resolveCharacter(rowU[wS]),Taxon.resolveCharacter(rowV[wS])};
					
					for(int i=0;i<chars.length;i++){
						switch(chars[i]){
							case 'g': gCount++; break;
							case 'c': cCount++; break;
							case 'a': aCount++; break;
							case 't': tCount++; break;
						}
					}
				}
			}
			
			// Looking accross 4 taxa i.e. X,Y,U,V
			gFrequency = (double)gCount/((double)windowSize*4.0);
			cFrequency = (double)cCount/((double)windowSize*4.0);
			aFrequency = (double)aCount/((double)windowSize*4.0);
			tFrequency = (double)tCount/((double)windowSize*4.0);
		}
		
		
		double c1 = 0;
		double c2 = 0;
		double c3 = 0;
		
		// Put all character sequences for the four taxa into a matrix...
		char[][] charMatrix = {rowX,rowY,rowU,rowV};
		double[] distances = new double[6];
		int distanceCounter=0;
		
		// Get all six possible pairwise distances from the 4 taxa...
		for (int a = 0; a < 4; a++) {
			for (int b = a+1; b < 4; b++) {
				distances[distanceCounter++]=delta(s,charMatrix[a],charMatrix[b]);
			}
		}
				
 		//distances input for tree topology ((1,2),(3,4));
 		quartetDistanceTree tree1 = new quartetDistanceTree(distances[0], distances[1], distances[2], distances[3], distances[4], distances[5]);

 		//distances input for tree topology ((1,3),(2,4));
 		quartetDistanceTree tree2 = new quartetDistanceTree(distances[1], distances[0], distances[2], distances[3], distances[5], distances[4]);

 		//distances input for tree topology ((1,4),(2,3));
 		quartetDistanceTree tree3 = new quartetDistanceTree(distances[2], distances[0], distances[1], distances[4], distances[5], distances[3]);

 		
 		double s1=0.0, s2=0.0, s3=0.0;
 		
 		if(USE_LEAST_SQUARED){
 	 		double wls1 = tree1.getwLS();
 			double wls2 = tree2.getwLS();
 			double wls3 = tree3.getwLS();
 			
 			 // least-squares
 			s1 = (1.0/wls1)/(1.0/wls1 + 1.0/wls2 + 1.0/wls3);
			s2 = (1.0/wls2)/(1.0/wls1 + 1.0/wls2 + 1.0/wls3);
			s3 = (1.0/wls3)/(1.0/wls1 + 1.0/wls2 + 1.0/wls3);
 		}
 		else{
			//minimum evolution
//			
	 		s1 = (1.0/tree1.getMe())/(1.0/tree1.getMe() + 1.0/tree2.getMe() + 1.0/tree3.getMe());
			s2 = (1.0/tree2.getMe())/(1.0/tree1.getMe() + 1.0/tree2.getMe() + 1.0/tree3.getMe());
			s3 = (1.0/tree3.getMe())/(1.0/tree1.getMe() + 1.0/tree2.getMe() + 1.0/tree3.getMe());
//			System.out.println("s1:" + s1);
/*
			double bme1 = tree1.getBme();	
			double bme2 = tree2.getBme();
			double bme3 = tree3.getBme();
		
 			s1 = (1.0/bme1)/(1.0/bme1 + 1.0/bme2 + 1.0/bme3);
			s2 = (1.0/bme2)/(1.0/bme1 + 1.0/bme2 + 1.0/bme3);
			s3 = (1.0/bme3)/(1.0/bme1 + 1.0/bme2 + 1.0/bme3);
*/ 		}
 		
//		System.out.println("Use LS:" + USE_LEAST_SQUARED + ": " + s1+"\t"+s2+"\t"+s3);
		
		// Now we have the counts, create the weights and the triplet
		return generateTriplet(s,s1,s2,s3);
/*
		Taxon[] taxon = VisRD.getAllTaxon();
		System.out.println(taxon[q.getX()].getLabel() + "," + taxon[q.getY()].getLabel() + "," + getDistanceBetweenTwoAlignments (rowX, rowY, s));
		return null;
*/	}
	
	protected double delta(int s, char[] rowOne, char[] rowTwo){
		int p1Count=0, p2Count=0, qCount=0;
		
		for (int wS = s - windowSize / 2; wS < s + windowSize / 2; wS++) {
			if ((! useMask) || (! mask[wS])) { // If position is not masked...
				// ... then, check whether the things equal or not...
				char charOne =Taxon.resolveCharacter(rowOne[wS]);
				char charTwo =Taxon.resolveCharacter(rowTwo[wS]);

				if(charOne!=charTwo){
					if((charOne=='a'&&charTwo=='g')||(charOne=='g'&&charTwo=='a')){ p1Count++; } // A-G transitions
					else if((charOne=='c'&&charTwo=='t')||(charOne=='t'&&charTwo=='c')){ p2Count++; } // C-T transitions
					else{ qCount++; } //transversions
				}
			}
		}

		//	windowSize=8007; // Testing...
		double p1 = (double)p1Count/(double)windowSize;
		double p2 = (double)p2Count/(double)windowSize;
		double q = (double)qCount/(double)windowSize;
		double p = p1+p2; // all transitions
		
		switch(type){
			case JC: return getJCdistance((p+q));
			case K2P: return getK2Pdistance(p,q);
			case F84: return getF84distance((p),(q),aFrequency, cFrequency, gFrequency, tFrequency);
			case TN: return getTNdistance(p1,p2,q,aFrequency, cFrequency, gFrequency, tFrequency);
		}
		
		return 0.0;
	}
	
    private static double getJCdistance(double p) {    	 
    	double distance = -(3.0/4.0)*Math.log(1.0 - (4.0/3.0)*p);
    	return distance;    	
    }

    private static double getJCgdistance(double p, double a) {    	 
    	double distance = (3.0/4.0)*a*(Math.pow((1.0 - (4.0/3.0)*p), (-1.0/a)) - 1.0);
    	return distance;    	
    }

    private static double getK2Pdistance(double p, double q) {    	 
    	double distance = -0.5*Math.log(1.0 - 2.0*p - q) - 0.25*Math.log(1.0 - 2.0*q);
    	return distance;    	
    }

    private static double getK2Pgdistance(double p, double q, double a) {    	 
    	double distance = 0.5*a*(Math.pow((1.0 - 2.0*p - q), (-1.0/a)) + 0.5*Math.pow((1.0 - 2.0*q), (-1.0/a)) - (3.0/2.0));
    	return distance;    	
    }

    //McGuire, Prentice and Wright    
    private static double getF84distance(double p, double q, double piA, double piC, double piG, double piT) {
 
    	double a = (piC*piT/(piC + piT)) + piA*piG/(piA + piG);
    	double b = piC*piT + piA*piG;
    	double c = (piA + piG)*(piC +piT);

    	return -2.0*a* Math.log(1.0-p/(2.0*a)-(a-b)*q/(2.0*a*c)) + 2.0*(a-b-c)*Math.log(1.0-q/(2.0*c));    	
    }

    private static double getF84gdistance(double p, double q, double piA, double piC, double piG, double piT, double a) {
    	 
    	double piR = piA + piG;
    	double piY = piC + piT;

    	double alpha1  = (2.0*(piC*piT + piA*piG) + 2.0*((piT*piC*piR)/piY + (piA*piG*piY)/piR)*(1.0 - q/(2.0*piY*piR)) - p)/((2.0*piC*piT)/piY + 2.0*piA*piG/piR);

    	double beta = 1.0 - q/(2.0*piY*piR);
    	
    	alpha1 = -(a * (1.0 - Math.pow(alpha1, (-1.0/a))));
    	beta = -(a*(1.0 - Math.pow(beta, (-1.0/a))));
    	
    	double alpha1t = 0.5 * alpha1;
    	double betat = 0.5 * beta;
    	double kappa = alpha1t/betat;
    	
    	double f84g = 4.0 * betat * (piC*piT*(1.0 + kappa/piY) + piA*piG*(1.0 + kappa/piR) + piY*piR);
    	
    	return f84g;
    }

    private static double getTNdistance(double p1, double p2, double q, double piA, double piC, double piG, double piT) {
    	
    	double piR = piA + piG;
    	double piY = piC + piT;
    	
    	double firstTerm = -((2.0*piA*piG)/piR)*Math.log(1.0 - (piR*p1)/(2.0*piA*piG) - q/(2.0*piR));
    	double secondTerm = -((2.0*piT*piC)/piY)*Math.log(1.0 - (piY*p2)/(2.0*piT*piC) - q/(2.0*piY));
    	double thirdTerm = -2.0*(piR*piY - (piA*piG*piY)/piR - (piT*piC*piR)/piY)*Math.log(1.0 - q/(2.0*piR*piY));
    	
    	double distance = firstTerm + secondTerm + thirdTerm;
    	
    	return distance;
    }

    private static double getTNgdistance(double p1, double p2, double q, double piA, double piC, double piG, double piT, double a) {
    	
    	double piR = piA + piG;
    	double piY = piC + piT;
    	
    	double firstTerm = ((piA*piG)/piR)*Math.pow((1.0 - (piR*p1)/(2.0*piA*piG) - q/(2.0*piR)), (-1.0/a));
    	double secondTerm = ((piT*piC)/piY)*Math.pow((1.0 - (piY*p2)/(2.0*piT*piC) - q/(2.0*piY)), (-1.0/a));
    	double thirdTerm = (piR*piY - (piA*piG*piY)/piR - (piT*piC*piR)/piY)*Math.pow((1.0 - q/(2.0*piR*piY)), (-1.0/a));
    	double fourthTerm = -piA*piG - piT*piC - piR*piY;
    	
    	double distance = 2.0*a*(firstTerm + secondTerm + thirdTerm + fourthTerm);
    	
    	return distance;
    }

/** from Rzhetsky  and Nei 1995, not sure if this implementation is correct   
    private static double getRzhetskyHKYdistance(double p1, double p2, double q, double piA, double piC, double piG, double piT) {
    	
    	double distance = 0.0;
    	
    	double piR = piA + piG;
    	double piY = piC + piT;
    	
    	double c = (1.0 - (q/(2.0*piR*piY)));
    	double e = (1.0 -q/(2.0*piR)-(piR*p1)/(2.0*piA*piG));
    	double f = (1.0 - (piY*p2)/(2.0*piC*piT) -q/(2.0*piY));
    	
    	double delta = Math.pow((2.0*Math.pow(piR, 2.0)*e), -1) - Math.pow((2.0*Math.pow(piR, 2)*c), -1);
    	double epsilon = Math.pow(2.0*piA*piG*e, -1);
    	double dzeta = Math.pow((2.0*Math.pow(piY, 2)*f), -1) - Math.pow((2.0*Math.pow(piY, 2)*c),-1);
    	double eta = Math.pow((2*piC*piT*f), -1);
       	double nu = Math.pow((2*piR*piY*c), -1);
       	
       	double va1 = ((Math.pow(delta, 2)*q + Math.pow(epsilon, 2)*p1) - Math.pow((delta*q+epsilon*p1), 2))/eta;
       	double va2 = ((Math.pow(dzeta, 2)*q + Math.pow(eta, 2)*p2) - Math.pow((dzeta*q+eta*p2), 2))/eta;
       	double cova1a2 = (delta*dzeta*q*(1.0-q) - delta*eta*q*p2 - epsilon*eta*p1*p2)/eta;
       	double cova1a3 = nu*q*(delta*(1.0 - q) - epsilon*p1)/eta;
       	double cova2a3 = nu*q*(dzeta*(1.0 - q) - eta*p2)/eta;
    	
       	double gamma = (va2 - cova1a2)/(va1+va2 - 2.0*cova1a2) + ((piR*piY)/(piA*piG+piC*piT))*((cova1a3 - cova2a3)/(va1 + va2 - 2.0*cova1a2));
    	    	
    	double a1 = (piY/piR)*Math.log(1.0 - q/(2.0*piR*piY)) - Math.log(1.0 - q/(2.0*piR) - (piR*p1)/(2.0*piA*piG))/piR;
    	double a2 = (piR/piY)*Math.log(1.0 - q/(2.0*piR*piY)) - Math.log(1.0 - q/(2.0*piY) - (piY*p2)/(2.0*piC*piT))/piY;
    	double a3 = -Math.log(1.0 - q/(2*piR*piY));
    	
    	distance = 2.0*(piA*piG + piC*piT)*(gamma*a1 + (1 - gamma)*a2) + 2*piR*piY*a3;
     	
    	return distance;
    } **/
}

class quartetDistanceTree {
	// pairwise matrix distances
	private double d12;
	private double d13;
	private double d14;
	private double d23;
	private double d24;
	private double d34;
	// pairwise patristic distances	
	private double p12;
	private double p13;
	private double p14;
	private double p23;
	private double p24;
	private double p34;
	
	//branches
	/**
	 * 
	 * 	1           3
	 * 	 \         /
	 * 	 a\   c  d/
	 * 	   -------
	 * 	 b/       \e
	 *   /         \
	 *  2           4
	 *
	 * 
	 */	
	private double a;
	private double b;
	private double c;
	private double d;
	private double e;
	
	public quartetDistanceTree() {
		d12 = d13 = d14 = d23 = d24 = d34 = 0.0;
		p12 = p13 = p14 = p23 = p24 = p34 = 0.0;
	}
	
	public quartetDistanceTree(double distance12, double distance13, double distance14, double distance23, double distance24, double distance34) {
		d12 = distance12;
		d13 = distance13;
		d14 = distance14;
		d23 = distance23;
		d24 = distance24;
		d34 = distance34;
		
		//least squares solution for branch lengths
		a =  0.5*d12 + 0.25*(d13-d23+d14-d24);
		b = d12 - a;
		c = 0.25*(d13 + d23 +d14 + d24) -0.5*(d12 + d34);
		d = 0.5*d34 + 0.25*(d13 + d23 - d14 -d24);
		e = d34 - d;
		
		p12 = a + b;
		p13 = a + c + d;
		p14 = a + c + e;
		p23 = b + c + d;
		p24 = b + c + e;
		p34 = d + e;
		
	}
    public double getwLS() {
        double ls;
        
        ls = (Math.pow((d12 - p12), 2)/Math.pow(d12, 2)) + (Math.pow((d13 - p13), 2)/Math.pow(d13, 2)) + (Math.pow((d14 - p14), 2)/Math.pow(d14, 2)) + (Math.pow((d23 - p23), 2)/Math.pow(d23, 2)) + (Math.pow((d24 - p24), 2)/Math.pow(d24,2)) + (Math.pow((d34 - p34), 2)/Math.pow(d34, 2));  
        
    	return ls;
    }
    public double getBme() {
        double bme;
        
        bme = (Math.pow((d12 - p12), 2)/Math.pow(2, p12)) + (Math.pow((d13 - p13), 2)/Math.pow(2, p13)) + (Math.pow((d14 - p14), 2)/Math.pow(2, p14)) + (Math.pow((d23 - p23), 2)/Math.pow(2, p23)) + (Math.pow((d24 - p24), 2)/Math.pow(2, p24)) + (Math.pow((d34 - p34), 2)/Math.pow(2, p34));  
        
    	return bme;
    }

    public double getMe() {
    	double me;
    	
    	double a_pos = a;
    	double b_pos = b;
    	double c_pos = c;
    	double d_pos = d;
    	double e_pos = e;
    	
    	
    	if (a < 0) {a_pos = 0.0;}
       	if (b < 0) {b_pos = 0.0;}
       	if (c < 0) {c_pos = 0.0;}
       	if (d < 0) {d_pos = 0.0;}
       	if (e < 0) {e_pos = 0.0;}
    	
    	me = a_pos + b_pos + c_pos + d_pos + e_pos;
    	
    	return me;
    	
    }    
}

class ProteinDistanceCalculator extends DistanceCalculator{
	private DistanceFunction df;
	
	public ProteinDistanceCalculator(Taxon[] taxon, DistanceFunction df){
		super(taxon, PROTEIN);
		this.df=df;
		System.out.println("Using corrected distances i.e. those using sigma^N");
	}
	
/*	public Triplet processWindow(int s, char[] rowX, char[] rowY, char[] rowU, char[] rowV, Quartet q){
		System.out.println("process window using the blosum matrix");
		return null;
	}
*/	
	protected double delta(int s, char[] rowOne, char[] rowTwo){
		double distanceOneTwo=0.0; // sigma(s1,s2)
		double distanceOneOne=0.0; // sigma(s1,s1)
		double distanceTwoTwo=0.0; // sigma(s2,s2)

		for (int wS = s - windowSize / 2; wS < s + windowSize / 2; wS++) {
			if ((! useMask) || (! mask[wS])) { // If position is not masked...
				distanceOneTwo+=df.delta(rowOne[wS],rowTwo[wS]);
				distanceOneOne+=df.delta(rowOne[wS],rowOne[wS]);
				distanceTwoTwo+=df.delta(rowTwo[wS],rowTwo[wS]);
			}
		}
		
		double expectedScore = df.getExpectedScore();
//		System.out.println("distanceOneTwo:" + distanceOneTwo);
		
		double sigmaN = distanceOneTwo - (expectedScore*windowSize);
		double sigmaUN = ((distanceOneOne + distanceTwoTwo)/2.0) - (expectedScore*windowSize);
	//	System.out.println("sigmaN:" + sigmaN);
//		System.out.println("sigmaUN:" + sigmaUN);
		
		double result = -Math.log(sigmaN/sigmaUN)*100.0;
		if(Double.isNaN(result)){
		//	System.out.println(distanceOneTwo + "-" + expectedScore + "*" + windowSize);
			
			System.out.print("Warning: can't log " + sigmaN + "/" + sigmaUN);
			
			
			StringBuilder alignmentOne = new StringBuilder();
			StringBuilder alignmentTwo = new StringBuilder();
			for (int wS = s - windowSize / 2; wS < s + windowSize / 2; wS++) {
				alignmentOne.append(rowOne[wS]);
				alignmentTwo.append(rowTwo[wS]);
				
			//	System.out.println(rowOne[wS] + " -> " + rowTwo[wS] + " is " + df.delta(rowOne[wS],rowTwo[wS]));
			}
			
			System.out.println(". In window " + (s - windowSize / 2) + " to " + (s + windowSize / 2) + ".");
			System.out.println("Remove ambiguous stretches or increase window size");
			
//			System.exit(0);
			
			return expectedScore*windowSize;
		}
	//	System.out.println("result:" + result);
		return result;
	}
}

/* All the distance functions */

interface DistanceFunction{
	public double delta(char a, char b);
	public double getExpectedScore();
}
/*
class AlmostKimuraDistanceFunction implements DistanceFunction{
	// This is for DNA and RNA alignments
	// b=10.0 a=5.0
	
	private static final String alphabet = new String ("atugc");
	private static final double[][] matrix = {new double[]
							{20.0, 10.0, 10.0, 5.0, 5.0},
								new double[]
							{10.0, 20.0, 20.0, 5.0, 5.0},
								new double[]
							{10.0, 20.0, 20.0, 5.0, 5.0},
								new double[]
							{5.0, 5.0, 5.0, 20.0, 10.0},
								new double[]
							{5.0, 5.0, 5.0, 10.0, 20.0}};

	public double delta(char a, char b){
		int x = alphabet.indexOf (a);
		int y = alphabet.indexOf (b);
		
		if(x > - 1 && y > - 1){ return matrix[x][y]; }
		else{ return 0.0; }
	}
}
*/
class BlosumDistanceFunction implements DistanceFunction{
	// This is for protein alignments
	private static String alphabet= new String ("arndcqeghilkmfpstwyv");
	private double[][] matrix = {new double[] 
		{4.0,-1.0,-2.0,-2.0, 0.0,-1.0,-1.0, 0.0,-2.0,-1.0,-1.0,-1.0,-1.0,-2.0,-1.0, 1.0, 0.0,-3.0,-2.0, 0.0},
								new double[] 
	   {-1.0, 5.0, 0.0,-2.0,-3.0, 1.0, 0.0,-2.0, 0.0,-3.0,-2.0, 2.0,-1.0,-3.0,-2.0,-1.0,-1.0,-3.0,-2.0,-3.0},
								new double[] 
	   {-2.0, 0.0, 6.0, 1.0,-3.0, 0.0, 0.0, 0.0, 1.0,-3.0,-3.0, 0.0,-2.0,-3.0,-2.0, 1.0, 0.0,-4.0,-2.0,-3.0},
								new double[] 
	   {-2.0,-2.0, 1.0, 6.0,-3.0, 0.0, 2.0,-1.0,-1.0,-3.0,-4.0,-1.0,-3.0,-3.0,-1.0, 0.0,-1.0,-4.0,-3.0,-3.0},
								new double[] 
		{0.0,-3.0,-3.0,-3.0, 9.0,-3.0,-4.0,-3.0,-3.0,-1.0,-1.0,-3.0,-1.0,-2.0,-3.0,-1.0,-1.0,-2.0,-2.0,-1.0},
								new double[] 
	   {-1.0, 1.0, 0.0, 0.0,-3.0, 5.0, 2.0,-2.0, 0.0,-3.0,-2.0, 1.0, 0.0,-3.0,-1.0, 0.0,-1.0,-2.0,-1.0,-2.0},
								new double[] 
	   {-1.0, 0.0, 0.0, 2.0,-4.0, 2.0, 5.0,-2.0, 0.0,-3.0,-3.0, 1.0,-2.0,-3.0,-1.0, 0.0,-1.0,-3.0,-2.0,-2.0},
								new double[] 
		{0.0,-2.0, 0.0,-1.0,-3.0,-2.0,-2.0, 6.0,-2.0,-4.0,-4.0,-2.0,-3.0,-3.0,-2.0, 0.0,-2.0,-2.0,-3.0,-3.0},
								new double[] 
	   {-2.0, 0.0, 1.0,-1.0,-3.0, 0.0, 0.0,-2.0, 8.0,-3.0,-3.0,-1.0,-2.0,-1.0,-2.0,-1.0,-2.0,-2.0, 2.0,-3.0},
								new double[] 
	   {-1.0,-3.0,-3.0,-3.0,-1.0,-3.0,-3.0,-4.0,-3.0, 4.0, 2.0,-3.0, 1.0, 0.0,-3.0,-2.0,-1.0,-3.0,-1.0, 3.0},
								new double[] 
	   {-1.0,-2.0,-3.0,-4.0,-1.0,-2.0,-3.0,-4.0,-3.0, 2.0, 4.0,-2.0, 0.0, 2.0,-3.0,-2.0,-1.0,-2.0,-1.0, 1.0},
								new double[] 
	   {-1.0, 2.0, 0.0,-1.0,-3.0, 1.0, 1.0,-2.0,-1.0,-3.0,-2.0, 5.0,-1.0,-3.0,-1.0, 0.0,-1.0,-3.0,-2.0,-2.0},
								new double[] 
	   {-1.0,-1.0,-2.0,-3.0,-1.0, 0.0,-2.0,-3.0,-2.0, 1.0, 2.0,-1.0, 5.0, 0.0,-2.0,-1.0,-1.0,-1.0,-1.0, 1.0},
								new double[] 
	   {-2.0,-3.0,-3.0,-3.0,-2.0,-3.0,-3.0,-3.0,-1.0, 0.0, 0.0,-3.0, 0.0, 6.0,-4.0,-2.0,-2.0, 1.0, 3.0,-1.0},
								new double[] 
	   {-1.0,-2.0,-2.0,-1.0,-3.0,-1.0,-1.0,-2.0,-2.0,-3.0,-3.0,-1.0,-2.0,-4.0, 7.0,-1.0,-1.0,-4.0,-3.0,-2.0},
								new double[] 
		{1.0,-1.0, 1.0, 0.0,-1.0, 0.0, 0.0, 0.0,-1.0,-2.0,-2.0, 0.0,-1.0,-2.0,-1.0, 4.0, 1.0,-3.0,-2.0,-2.0},
								new double[] 
		{0.0,-1.0, 0.0,-1.0,-1.0,-1.0,-1.0,-2.0,-2.0,-1.0,-1.0,-1.0,-1.0,-2.0,-1.0, 1.0, 5.0,-2.0,-2.0, 0.0},
								new double[] 
	   {-3.0,-3.0,-4.0,-4.0,-2.0,-2.0,-3.0,-2.0,-2.0,-3.0,-2.0,-3.0,-1.0, 1.0,-4.0,-3.0,-2.0,11.0, 2.0,-3.0},
								new double[] 
	   {-2.0,-2.0,-2.0,-3.0,-2.0,-1.0,-2.0,-3.0, 2.0,-1.0,-1.0,-2.0,-1.0, 3.0,-3.0,-2.0,-2.0, 2.0, 7.0,-1.0},
								new double[] 
		{0.0,-3.0,-3.0,-3.0,-1.0,-2.0,-2.0,-3.0,-3.0, 3.0, 1.0,-2.0, 1.0,-1.0,-2.0,-2.0, 0.0,-3.0,-1.0, 4.0}};

	public BlosumDistanceFunction(){
	/*	for(int u=0;u<20;u++){
			for(int v=0;v<20;v++){
				matrix[u][v] = Math.pow (10, matrix[u][v] / 10);
			}
		}
	*/
	}
	
	public double getExpectedScore(){
		double average=0.0;
		for(int i=0;i<matrix.length;i++){
			for(int j=0;j<matrix[i].length;j++){
				average+=matrix[i][j];
			}
		}
		
		double expectedScore = average/(matrix.length*matrix.length);
		return expectedScore;
	}
	
	public double delta(char a, char b){
		int x = alphabet.indexOf (a);
		int y = alphabet.indexOf (b);
		
		if (x > - 1 && y > - 1) { return matrix[x][y]; }
		else { return 0.0; }
	}
}

class HammingDistanceFunction implements DistanceFunction{
	public double delta (char a, char b){
		if(a==b){ return 0.0; }
		else { return 1.0; }
	}
	
	public double getExpectedScore(){ return 0.0; } 
}


class KimuraDistanceFunction implements DistanceFunction{
	// This is Kimura's "2-parameter" model for DNA and RNA alignments
	private static final String alphabet = new String ("agctu");
	private static double[][] matrix;
	private static double[][] fastMatrix;

	public static void setParameters(double a){
		double b = 1.0;
		double leadingDiagonal = 1.0-a-2.0*b;
/*		
		double[][] m = {new double[]
				{leadingDiagonal, a, b, b, b},
					new double[]
				{a, leadingDiagonal, b, b, b},
					new double[]
				{b, b, leadingDiagonal, a, a},
					new double[]
				{b, b, a, leadingDiagonal, leadingDiagonal},
					new double[]
				{b, b, a, leadingDiagonal, leadingDiagonal}};
		
		// hack
		matrix=m;
*/
		
		fastMatrix = new double[120][120]; // Large in terms of space, fast in terms of time!
		fastMatrix['a']['a']=leadingDiagonal;
		fastMatrix['c']['c']=leadingDiagonal;
		fastMatrix['g']['g']=leadingDiagonal;
		fastMatrix['t']['t']=leadingDiagonal;
		
		fastMatrix['a']['g']=a;
		fastMatrix['a']['c']=b;
		fastMatrix['a']['t']=b;
		fastMatrix['a']['u']=b;
		
		fastMatrix['g']['a']=a;
		fastMatrix['g']['c']=b;
		fastMatrix['g']['t']=b;
		fastMatrix['g']['u']=b;
		
		fastMatrix['c']['a']=b;
		fastMatrix['c']['g']=b;
		fastMatrix['c']['t']=a;
		fastMatrix['c']['u']=a;
		
		fastMatrix['t']['a']=b;
		fastMatrix['t']['g']=b;
		fastMatrix['t']['c']=a;
		fastMatrix['t']['u']=leadingDiagonal;
		
		fastMatrix['u']=fastMatrix['t'];
	}
/*
	public double delta(char a, char b){
		int x = alphabet.indexOf (a);
		int y = alphabet.indexOf (b);
		
		if(x > - 1 && y > - 1){ return matrix[x][y]; }
		else{ return 0.0; }
	}
*/	
	public double delta(char a, char b){ return fastMatrix[a][b]; }
	public double getExpectedScore(){ return 0.0; }
}


/*
class JCDistanceFunction implements DistanceFunction{
	// This is Kimura's "2-parameter" model for DNA and RNA alignments
	private static final String alphabet = new String ("agctu");
	private static double[][] matrix;
	private static double[][] fastMatrix;
	private static double oneOverLambda;

	public static void setParameters(double aFrequency,double cFrequency,double gFrequency,double tFrequency){
		double leadingDiagonal=0.0;
		fastMatrix = new double[120][120]; // Large in terms of space, fast in terms of time!
		fastMatrix['a']['a']=leadingDiagonal;
		fastMatrix['c']['c']=leadingDiagonal;
		fastMatrix['g']['g']=leadingDiagonal;
		fastMatrix['t']['t']=leadingDiagonal;
		
		fastMatrix['a']['g']=a;
		fastMatrix['a']['c']=b;
		fastMatrix['a']['t']=b;
		fastMatrix['a']['u']=b;
		
		fastMatrix['g']['a']=a;
		fastMatrix['g']['c']=b;
		fastMatrix['g']['t']=b;
		fastMatrix['g']['u']=b;
		
		fastMatrix['c']['a']=b;
		fastMatrix['c']['g']=b;
		fastMatrix['c']['t']=a;
		fastMatrix['c']['u']=a;
		
		fastMatrix['t']['a']=b;
		fastMatrix['t']['g']=b;
		fastMatrix['t']['c']=a;
		fastMatrix['t']['u']=leadingDiagonal;
		
		fastMatrix['u']=fastMatrix['t'];
	}
	public double delta(char a, char b){ return oneOverLambda*fastMatrix[a][b]; }
}
*/

class ProteinMatrix implements DistanceFunction{
	private char[] characterMap;
	private static int[][] matrix;
	
	public static void main(String[] args){
		new ProteinMatrix("Between10%");
	}
	
	public ProteinMatrix(String filename){
		matrix = new int[124][124];
		
		try{
			java.io.BufferedReader br = new java.io.BufferedReader(new java.io.FileReader(filename));
			
			// Character map tells us which position the different protein characters are in...
			characterMap = new char[24];
			int rowNumber=-1;
			
			String line = br.readLine();
			while(line!=null){
				if(line.charAt(0)!='#'){ //This is not a comment line
				     java.util.StringTokenizer st = new java.util.StringTokenizer(line.toLowerCase());

				     if(rowNumber==-1){
				     	int i=0;
				     	
				        while(st.hasMoreTokens()) {
							characterMap[i++] = st.nextToken().charAt(0);
						}
				     }
				     else{
				     	int colNumber=0;
				     	st.nextToken(); // Ignore first token
				     		
				     	while(st.hasMoreTokens()) {
				     		matrix[characterMap[rowNumber]][characterMap[colNumber]] = Integer.parseInt(st.nextToken());
				     		colNumber++;
						}
				     }
				     
				     rowNumber++;
				}
				
			//	System.out.println(line);
				line = br.readLine();
			}
		}
		catch(Exception e){
			System.out.println("Exception whilst trying to read protein matrix!");
			printMatrix();
			e.printStackTrace();
		}
	}
	
	public double getExpectedScore(){
		int n = 20;
		if(matrix.length<n){ n=matrix.length; }

		double average=0.0;
		for(int i=0;i<n;i++){
			for(int j=0;j<n;j++){
				average+=matrix[characterMap[i]][characterMap[j]];
			}
		}
		
		double expectedValue = average/(n*n);
		return expectedValue;
	}

	public double delta(char a, char b){ return matrix[a][b]; }
	
	private void printMatrix(){
		for(int i=0;i<characterMap.length;i++){
			System.out.print(characterMap[i] + ": ");
			for(int j=0;j<characterMap.length;j++){
				System.out.print(matrix[characterMap[i]][characterMap[j]]);
			}
			System.out.println();
		}
	}
}