package visrd;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.ArrayList;
import java.io.*;

/**

	The VisRD GUI

	@author Kristoffer Forslund
	@version 2.0, 07/06/03

	This class is the graphics user interface for the VisRD package.
	It will receive taxa and characters (sequence data) from the
	calling HighWay qlippoth object (shell object, the interface
	between jSplits proper and VisRD) and also, if data is changed
	in jSplits, it will receive a signal enabling the user to
	retrieve the data from jSplits anew.

	Actual analysis is done from VisRD objects, settings are kept in
	the main VisRD class and output is done via HighwayPlot, OccupancyPlot,
	QuartetMap and FrequencyDisplay objects. These could in turn
	be replaced by the Tuebingen groups metaviewer, if it
	is incorporated into jSplits. At present, most of these support saving
	to .eps files.

*/

public class VisRDGUI extends JFrame implements ActionListener{
	private static JMenuItem scanItem, opItem, hwItem, qmItem, trItem;
	private static JCheckBoxMenuItem hwStepwise;
	
	/** Actual scan data */

	private static QuartetSet data;
    private CloseableTabbedPane tabbedStructure;
    
//	private JScrollPane mP;
//	private static JTextArea mList = new JTextArea();
	private JMenuItem aboutItem, dumpItem, loadItem, exitItem;
	
	private static TaxonSelectorPanel taxonSelector;
	private static ParametersPanel parametersPanel;
	
    private static Cursor waitCursor = new Cursor(Cursor.WAIT_CURSOR);

	public VisRDGUI (){
		Taxon[] taxon = VisRD.getAllTaxon();
		/** Store stuff into local objects (or rather local pointers... keep them with this object, anyway)
			INHT */

		VisRD.setSequenceLength (taxon[0].getNchar ()); // all taxon should have same sequence length

		/**

			Make sure that when the window closes, the HighWay object will know, and so
			be able to restart the GUI when necessary

		*/

        JMenuBar menuBar = new JMenuBar ();
        JMenu fileMenu = new JMenu ("File");

		aboutItem = new JMenuItem("About");
		aboutItem.addActionListener(this);
        fileMenu.add (aboutItem);
        
        dumpItem = new JMenuItem("Dump character data");
		dumpItem.addActionListener (this);
        fileMenu.add (dumpItem);
        
        loadItem = new JMenuItem("Load");
		loadItem.addActionListener (this);
        fileMenu.add (loadItem);
        
        exitItem = new JMenuItem("Exit");
        exitItem.addActionListener(this);
        fileMenu.add (exitItem);
                
        JMenu runMenu = new JMenu ("Run");
        
        scanItem = new JMenuItem ("Scan...");
		hwItem = new JMenuItem ("Highway Plot");
		hwStepwise = new JCheckBoxMenuItem("Use stepwise color");
		opItem = new JMenuItem ("Occupancy Plot");
		qmItem = new JMenuItem ("Quartet Mapping");
		trItem = new JMenuItem ("Taxon Ranking");
        
        scanItem.addActionListener (this);
		hwItem.addActionListener (this);
		hwStepwise.addActionListener (this);
		opItem.addActionListener (this);
		qmItem.addActionListener (this);
		trItem.addActionListener (this);

        hwItem.setEnabled (false);
        hwStepwise.setEnabled(false);
		opItem.setEnabled (false);
		qmItem.setEnabled (false);
		trItem.setEnabled (false);
        
        runMenu.add (scanItem);
        runMenu.addSeparator();
		runMenu.add (hwItem);
//		runMenu.add(hwStepwise); // Don't offer stepwise coloring, we can't reproduce previous reuslts anyway now....
		runMenu.addSeparator();
		runMenu.add (opItem);
		runMenu.addSeparator();
		runMenu.add (qmItem);
		runMenu.addSeparator();
		runMenu.add (trItem);
        
        menuBar.add (fileMenu);
        menuBar.add (runMenu);

        setJMenuBar (menuBar);

//		setTitle ("untitled");

		/**

			Tabs, to make fitting everything in easier

		*/

		tabbedStructure = new CloseableTabbedPane ();
		((Container) getContentPane ()).add (tabbedStructure);

		// Add a message tab

//		mList = new JTextArea ();

/*		mP = new JScrollPane (mList,
										  JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, 
										  JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
*/		
		parametersPanel = new ParametersPanel (taxon);
	
		taxonSelector = new TaxonSelectorPanel();
		taxonSelector.init (taxon);
		
		JScrollPane taxonSelectorPane = new JScrollPane (taxonSelector);
		tabbedStructure.addTab ("Quartet selection", taxonSelectorPane,false);

		JScrollPane paramsPane = new JScrollPane (parametersPanel);
		tabbedStructure.addTab ("Method parameters", paramsPane,false);
		
//		tabbedStructure.setCloseIconVisibleAt(1,);
//		tabbedStructure.addTab ("Messages", mP);

        setSize (850, 650);

//		mP.revalidate ();
		paramsPane.revalidate ();
		taxonSelectorPane.revalidate ();
		tabbedStructure.revalidate ();
		
		if(VisRD.getShowGUI()){ setVisible (true); }
	}
	
    public void actionPerformed (ActionEvent e){
    	if(e.getSource()==aboutItem){
			JOptionPane.showMessageDialog(null,"This verion of VisRD was written by Martin Lott and is an update of the version written by Kristoffer Forslund. Contributions by Philippe Lemmy and Vincent Moulton.");
    	}
    	else if(e.getSource()==dumpItem){ IOHandler.dumpCharacterData(VisRD.getAllTaxon()); }
    	else if(e.getSource()==loadItem){ doLoadPopup(); }
    	else if(e.getSource()==exitItem){ System.exit(0); }
    	else if(e.getSource()==hwItem){ highwayPlot(); }
    	else if(e.getSource()==hwStepwise){ VisRD.setUseStepwise(hwStepwise.isSelected()); }
    	else if(e.getSource()==qmItem){ quartetMap(); }
    	else if(e.getSource()==opItem){ occupancyPlot(); }
    	else if(e.getSource()==trItem){ taxonRanking(); }
    	else if(e.getSource()==scanItem){ scan(); }
    }
    
    private void doLoadPopup(){
    	File lastOpenFile = new File (System.getProperty ("user.dir"));
		File file = null;
	
		JFileChooser chooser = new JFileChooser (lastOpenFile);
 		chooser.addChoosableFileFilter (new FastaFilter());
		chooser.addChoosableFileFilter (new NexusFilter());
		            
		if (chooser.showOpenDialog (null) == JFileChooser.APPROVE_OPTION) {
			file = chooser.getSelectedFile ();
			lastOpenFile = file;
		}

		if (file == null || file.isFile () == false) {
			// problem loading
			displayError ("Error: Could not load requested file!");
		}
		else {
			try {
				System.out.println("Loading file " + file.getName () + "...");
				
				IOHandler.loadFile(file);
                        
				updateDataset ();

				System.out.println ("File loaded.");
				
				setTitle (file.getName ());
			}
			// TODO: Do something if the user cancelled, we removed CancelledException
			// try something like:  if(file == null||file.isFile ()==false){ throw new Exception ("Loading was cancelled!"); }
			catch (Exception E) {
				displayError ("Error: Error while reading file!");
			}
		}
    }

	public void setTitle(String fileName){
		super.setTitle(VisRD.PROGRAMNAME + " " + VisRD.VERSION + " Analysis of " + fileName);
	}

	public static void updateDatasets(){ //Total hack!!
		Taxon[] taxon = VisRD.getAllTaxon();
		VisRD.setSequenceLength (taxon[0].getNchar ()); // all taxon should have same sequence length

		if(!NexusIO.containsMultipleDataSets()){
			if(taxonSelector!=null){ taxonSelector.init (taxon); }
			if(parametersPanel!=null){ parametersPanel.update (taxon); }
		}
	}
	
	public void updateDataset(){ //Reload stuff
		hwItem.setEnabled (false);
		hwStepwise.setEnabled(false);
		opItem.setEnabled (false);
		trItem.setEnabled (false);
		qmItem.setEnabled (false);
		
		updateDatasets();
	}

	public void scan(){ // Scan method... This is the actual analysis step
		if(ScanWindow.thisWindow==null){ ScanWindow.thisWindow = new ScanWindow(VisRD.getAllTaxon(), null); }
		else{ ScanWindow.thisWindow.show(null); }
	}
	
	public void harvestScanParameters(){
		// Called by the VisRD scan thread...
		taxonSelector.harvest ();
		taxonSelector.harvestParameters();
		parametersPanel.harvest ();
	}

    public static void handleData (QuartetSet newData){
        data = newData;

        if (VisRD.getShowGUI()&&data.size()>0) {
			hwItem.setEnabled (true);
			hwStepwise.setEnabled(true);
			opItem.setEnabled (true);
			trItem.setEnabled (true);
			qmItem.setEnabled (true);
        }
    }
    
    // Display some warning to the user
    public static void displayError(String message){
    	displayMessage(message, JOptionPane.ERROR_MESSAGE);
    }
    
    public static void displayMessage(String message, int messageType){
    	 if(VisRD.getShowGUI()){ JOptionPane.showMessageDialog(null, message, VisRD.getFullProgramName(), messageType); }
    	 else{ System.out.println("Message:" + message); }
    	 
    }

	public void quartetMap (){ // Display quartet map data
		taxonSelector.harvestParameters();
		parametersPanel.harvest ();

		QuartetMap aQM = new QuartetMap (data, tabbedStructure);
        tabbedStructure.add ("Quartet Mapping", aQM);
        tabbedStructure.setSelectedComponent (aQM);
	}

	public void occupancyPlot (){ // Display occupancy plot
		taxonSelector.harvestParameters();
		parametersPanel.harvest ();

		OccupancyPlot aOP = new OccupancyPlot (data, tabbedStructure);
        tabbedStructure.add ("Occupancy Plot", aOP);
        tabbedStructure.setSelectedComponent (aOP);
	}

	public void highwayPlot (){ // Display Highway plot
		taxonSelector.harvestParameters();
		parametersPanel.harvest ();

		HighwayPlot aHP = new HighwayPlot (data, tabbedStructure);
        tabbedStructure.add ("Highway Plot", aHP);
        tabbedStructure.setSelectedComponent (aHP);
	}

	public void taxonRanking(){ // Display taxon rankings
		taxonSelector.harvestParameters();
		parametersPanel.harvest ();

		TaxonRankingPanel trp = new TaxonRankingPanel (data, VisRD.getAllTaxon());
        tabbedStructure.add("Taxon Ranking", trp);
        tabbedStructure.setSelectedComponent (trp);
	}
	
	public static RankingResults getRankingResults(){
		return TaxonRankingPanel.getRankingResults(data, VisRD.getAllTaxon(), false);
	}

    public void lock(){
        setCursor (waitCursor);
        scanItem.setEnabled (false);
    }

    public void unlock(){
        scanItem.setEnabled (true);
        setCursor (Cursor.getDefaultCursor ());
    }
}