/**

	VisRD main analysis class

	@version $Id: VisRD.java,v 1.1 2004/03/01 21:19:29 huson Exp $

	@author Kristoffer Forslund

	6.2003

*/

package visrd;

import javax.swing.*;
import java.util.ArrayList;
import java.io.File;
import java.io.FileReader;

/**

	This is the class which, given a data object and a settings object
	from either command-line arguments to something like HighWay above
	or from the VisRDGUI window will generate any one of the desired plot
	objects. Basically it can be given choice of settings and data
	and thats it - it should return something like a VisRDPlot interface.

	Alerting to changes in data should mean; in GUI mode, that the user may
	ask this class to resupply all plots from new data and same settings,
	or, in command-line mode as specified from the objects above, that
	it is made to do this again.

	This class takes window and step length, choice of generation method
	and parameters for quartet generation from settings object. It takes
	sequences from data block, caring for jSplits masking as it does.

	Then it generates the quartets in question. It does the analysis with them,
	and as it does, stores the coordinates at each step as well as records
	min/max and the like. Quartets are change-ordered in this stage. The end result
	is a QuartetList and a TrajectoryList using the same ordering.

	HighwayPlot objects take a QuartetList + TrajectoryList and plot in
	itself as a Highway plot. Same goes for Occupancy and animated quartet
	mappings.

	So, this is really just taking generating a quartet list
	and then providing an ordering and trajectory list for it.

*/

public class VisRD extends Thread
{
	public static final String PROGRAMNAME="VisRD", VERSION="3.0";
	
	private static String currentFilename, currentPathFilename;
	private static int displayQuartets = 10, stepSize = 10, windowSize = 200, plotType = 1, firstWindow, lastWindow, desiredQuartets, varianceType, sequenceLength;
	private static boolean useRandomQuartets, clusterQuartets, useStepwise, outputQuartetsToFile;
	private static boolean stepSizeSpecified, windowSizeSpecified, showGUI;
	private static double deltaThreshold;
	private static int[] includeTaxa, forceTaxa;
	
	private static int[][] taxaGroup;
	private static Object[] groupNames;
	
    private static VisRDGUI GUI;
    private QuartetSet result;
 
 	private static Taxon[] taxon;
 	
 	private static int nDatasets, shuffleChoice, datasetCounter;

	public VisRD (){}
	public static String getFullProgramName(){ return PROGRAMNAME + " " + VERSION; }

	public void run (){
		// First get the parameters...
		if(GUI!=null){ GUI.harvestScanParameters(); }
		Quartet[] quartets=null;

		try {
			if(ScanWindow.thisWindow!=null){ ScanWindow.setStatus("Generating quartets"); }
			
			quartets = QuartetGenerator.generateSet();
			
			if(ScanWindow.thisWindow!=null){ ScanWindow.setNumberOfQuartets (quartets.length);}
			else{
				System.out.println("Processing dataset " + datasetCounter + " of " + nDatasets + ". Generating " + quartets.length + " quartets");
			}
		}
		catch (java.lang.OutOfMemoryError e) {
			VisRDGUI.displayError ("Not enough memory error!\nThis may be solved by restarting with increased heap size (via the -Xmx switch).\nAlternately, reduce the number of quartets that are scanned by either \nexcluding sequences or sampling a smaller number of quartets.");
		}

		if (quartets==null||quartets.length==0){
			VisRDGUI.displayError ("VisRD: No taxa selected, scan aborted.");
		}

		if (lastWindow - firstWindow < windowSize) {
			VisRDGUI.displayError ("VisRD: Alignment too short for window size, scan aborted.");
		}

        else {

            // Generate quartets...
            result = new QuartetSet (quartets);

            /**

                Go through result, for each quartet do the mathz, that
                is, perform the computations window by window.
            */

            // Here, there should is selection as for which type of calculator to use...
            TrajectoryCalculator calculator;
            
            switch(ScanWindow.getMethodType()){
            	case 0:
            		calculator = new StatisticalGeometryCalculator (taxon);
            	break;
/*            	case 1:
            		calculator = new MaximumParsimonyCalculator (taxon);
            	break;
*/            	case 1:
            		calculator = new WeightedStatisticalGeometryCalculator (taxon, new HammingDistanceFunction ());
                break;
                case 2:
                	if(Taxon.isDNAorRNA()){ // Jukes Cantor
                		calculator = new DistanceCalculator (taxon,DistanceCalculator.JC);
                	}
                	else{
                		calculator = new WeightedStatisticalGeometryCalculator (taxon, new BlosumDistanceFunction ());
                	}
                break;
				default:
					String[] fileName = {"Between10%","Between38%","Within5%","PIVlogOdds"};
					if(Taxon.isDNAorRNA()){ // Kimura 2P
						calculator = new DistanceCalculator (taxon,ScanWindow.getMethodType());
                	}
                	else{
                		if(ScanWindow.getMethodType()==3){
							calculator = new ProteinDistanceCalculator (taxon,new BlosumDistanceFunction());
						}
						else{
							int index = ScanWindow.getMethodType()-4;
							if(index<fileName.length){
								calculator = new WeightedStatisticalGeometryCalculator (taxon, new ProteinMatrix(fileName[index]));	
							}
							else{
								calculator = new ProteinDistanceCalculator(taxon, new ProteinMatrix(fileName[index - fileName.length]));	
							}
						}
                	}
            }
            
           	System.out.println("START method" + ScanWindow.getMethodType ());


            Quartet q;
            Triplet[] t;

			try {
				if(ScanWindow.thisWindow!=null){
					ScanWindow.setStatus ("Calculating trajectories");
					ScanWindow.updateProgressMax(result.size());
				}
				else{
					System.out.println("Calculating trajectories");
				}
	
				int interval = (int)Math.floor((double)result.size ()/200.0);
				int errors = 0;
	
	            for (int n = 0; n < result.size (); n++) {
	
	                q = result.getQuartet (n);
	
	                /**
	
	                    Give rise to trajectorylist t from q
	                    and the chars
	
	                */
	                
	                if (interval > 0) {
	                
	                	if (n % interval == interval-1) {
	                	
	                		if(ScanWindow.thisWindow!=null){ ScanWindow.updateProgress(n+1); }
	                		else{ System.out.println("progess:" + (n+1) + " out of " + result.size ()); }
	                	//	GUI.message ((n+1) + " of " + result.size() + " trajectories calculated.");
	                		
	                	}
	                	
	                }
	
	                t = calculator.calculate (q);
	
					if(calculator.getWarning ()){ errors++; }
	
	           //     result.setTrajectory (n, t);

		           if(displayQuartets!=0){
		           		q.setTrajectoryList(t);
		           		q.calculateVariance();
		           }
		           else{
		          		q.calculateVariance(t);
		           }
	            }
/*	
				if (errors > 0) {
					VisRDGUI.displayError ("Warning: " + errors + " trajectories contain non-informative windows!");
				}
*/	
	            /**
	
	                Ask result to calculate variances
	
	            */
/*	
				if(sw!=null){ ScanWindow.setStatus ("Calculating variances"); }
				else{ System.out.println("Calculating variances"); }

				result.calculateVariances(GUI, varianceType);
*/
	            // Ask result to sort by variances
	
				if(ScanWindow.thisWindow!=null){ ScanWindow.setStatus ("Sorting quartets"); }
				else{ System.out.println("Sorting quartets"); }
				
	            result.sort ();
	
				if(ScanWindow.thisWindow!=null){
					ScanWindow.setStatus ("Scan completed"); 
					ScanWindow.updateProgress(result.size());
				}
				else{ System.out.println("Scan completed"); }
				
				VisRDGUI.handleData (result);

				if(VisRD.getShowGUI()&&!NexusIO.containsMultipleDataSets()&&!ScanWindow.containsMoreDataSets()){ ScanWindow.thisWindow.dispose(); }
				NexusIO.informScanComplete();
				VisRD.informScanComplete();
			}
			catch (java.lang.OutOfMemoryError e) {
				e.printStackTrace();
				VisRDGUI.displayError ("Not enough memory error!\nThis may be solved by restarting with increased heap size (via the -Xmx switch).\nAlternately, reduce the number of quartets that are scanned by either \nexcluding sequences or sampling a smaller number of quartets.");
			}
       }

	}
	
	/* All the settings... */
	
	public static boolean getShowGUI(){ return showGUI; }
	
	public static void setCurrentFile(File f){
	    String s = f.getName();
		int i = s.lastIndexOf('.');
	    if (i <1){ i=s.length(); }
	    
	    currentFilename = s.substring(0,i).toLowerCase();
		currentPathFilename = f.getPath().substring(0,f.getPath().lastIndexOf('.'));
		
		System.out.println("Set current file" + currentPathFilename);
	}
	
	public static String getCurrentFilename(){ return currentFilename; }
	public static String getCurrentPathFilename(){ return currentPathFilename; }
	
	public static boolean getClusterQuartets (){ return clusterQuartets; }
	public static void setClusterQuartets (boolean newClusterQuartets){clusterQuartets = newClusterQuartets; }
	
	public static int getDisplayQuartets (){ return displayQuartets; }
	public static void setDisplayQuartets(int newDisplayQuartets){ displayQuartets = newDisplayQuartets; }

	public static int getDesiredQuartets (){ return desiredQuartets; }
	public static void setDesiredQuartets (int newDesiredQuartets){ desiredQuartets = newDesiredQuartets; }
	
	public static boolean getUseRandomQuartets () { return useRandomQuartets; }
	public static void setUseRandomQuartets (boolean newUseRandomQuartets){useRandomQuartets = newUseRandomQuartets; }
	
	public static boolean getUseStepwise (){ return useStepwise; }
	public static void setUseStepwise (boolean newUseStepwise){ useStepwise = newUseStepwise; }


	public static int getWindowSize (){ return windowSize; }
	public static void setWindowSize(int newWindowSize){ windowSize = newWindowSize; }

	public static int getFirstWindow (){ return firstWindow; }
	public static void setFirstWindow (int newFirstWindow){ firstWindow = newFirstWindow; }
	
	public static int getLastWindow (){ return lastWindow; }
	public static void setLastWindow (int newLastWindow){ lastWindow = newLastWindow; }	

	public static void setTaxon(Taxon[] newTaxon){ taxon=newTaxon;	}

	public static Taxon[] getAllTaxon(){ return taxon; }
	public static int[][] getTaxaMatrix(){ return taxaGroup; }

	public static int[] getTaxaGroup(int i){ return taxaGroup[i]; }
	public static void setTaxaGroup(int i, int[] taxa){ taxaGroup[i]=taxa; }
	
	public static boolean getOutputQuartets(){ return outputQuartetsToFile; }
	public static void setOutputQuartets(boolean b){ outputQuartetsToFile=b; }
	
	public static void setupTaxaGroups(int size){ taxaGroup = new int[size][]; }

	public static int[] getIncludeTaxa (){ return includeTaxa; }
	public static void setIncludeTaxa (int[] newIncludeTaxa){ includeTaxa = newIncludeTaxa; }

	public static int[] getForceTaxa (){ return forceTaxa; }
	public static void setForceTaxa (int[] newForceTaxa){ forceTaxa = newForceTaxa; }

	public static int getPlotType (){ return plotType; }
	public static void setPlotType(int newPlotType){ plotType = newPlotType; }
	
	public static int getStepSize (){ return stepSize; }
	public static void setStepSize(int newStepSize){ stepSize = newStepSize; }

	public static double getDeltaThreshold (){ return deltaThreshold; }
	public static void setDeltaThreshold (double newDeltaThreshold){ deltaThreshold = newDeltaThreshold; }

	public static int getVarianceType (){ return varianceType; }
	public static void setVarianceType (int newVarianceType){ varianceType = newVarianceType; }
	
	public static int getSequenceLength (){ return sequenceLength; }
	public static void setSequenceLength (int newSequenceLength){ sequenceLength = newSequenceLength; setDefaultWindowSize(); }

	public static void setGroupNames(Object[] newGroupNames){ groupNames=newGroupNames; }
	public static Object[] getGroupNames(){ return groupNames; /* INDEX FROM 0 NOT 1 */ }
	
	public static void setDatasetCounter(int newDatasetCounter){ datasetCounter = newDatasetCounter; }
	public static int getDatasetCounter(){ return datasetCounter; }
	
	public static void scan(int newShuffleChoice, boolean outputToFile, int newnDatasets){
		shuffleChoice=newShuffleChoice;
		nDatasets=newnDatasets;
		
		Taxon[] taxa = VisRD.getAllTaxon();
		for(Taxon t: taxa){
			t.preserveMatrix();
		}
		NexusIO.resetRankingResults();
		
		Permutator.setPrintToFile(outputToFile);

		
//		if(shuffleChoice==0){
			informScanComplete(); // Start scanning...
//		}
//		else{
/*			if(outputToFile){
//				Thread myThread = new Thread(){
//					public void run(){
						if(VisRD.getShowGUI()){
							ScanWindow.thisWindow.setProgressBarMax(ScanWindow.thisWindow.getNdatasets());
							ScanWindow.thisWindow.setStatus("Writing datasets");
						}
						
						String fileName=null;
						
						if(shuffleChoice==1){
							fileName = Permutator.premuteSequences(VisRD.getAllTaxon(),nDatasets,false,true);
						}
						else if(shuffleChoice==2){
							fileName = Permutator.premuteSequences(VisRD.getAllTaxon(),nDatasets,true,true);
						}
		//			}
		//		};

			//	myThread.start();
			}
			else{
*/			//	informScanComplete(outputToFile); // Start scanning...
//			}
//		}
	}

	public static void informScanComplete(){
		// Grab result to be printed
	//	if(shuffleChoice!=0&&datasetCounter!=1){ NexusIO.addLatestRankingResult(); } //Ensure we have some results

		if(datasetCounter<=(nDatasets)){
			System.out.println(datasetCounter + " of " + nDatasets);
			// First do the shuffling
			if(shuffleChoice==1){
				Permutator.premuteSequences(VisRD.getAllTaxon(),datasetCounter,false);
			}
			else if(shuffleChoice==2){
				Permutator.premuteSequences(VisRD.getAllTaxon(),datasetCounter,true);
			}

			datasetCounter++;
			
			VisRD aVisRD = new VisRD ();
	
			try { // run VisRD analysis
				aVisRD.start ();
			}
			catch (Exception e) {
				e.printStackTrace();
				VisRDGUI.displayError ("Error while scanning: " + e.getMessage ());
			}
		}
		else{
			System.out.println("Scan complete, " + nDatasets + " datasets processed");
			// If file doesn't contain multiple datasets we know we're at the end based on the counter
			if(!NexusIO.containsMultipleDataSets()){
				NexusIO.saveRankingResults();
			}
			
			if(shuffleChoice!=0){
				Taxon[] taxa = VisRD.getAllTaxon();
				for(Taxon t: taxa){
					t.restoreMatrix();
				}
			}
		}
	}
	
	public static Object[] getLabelSet(){
		Object[] labelSet;
		if(clusterQuartets){
			labelSet = new Object[groupNames.length];
			for(int i=0;i<groupNames.length;i++){
				labelSet[i]=groupNames[i];
			}
		}
		else{
			labelSet = new Object[taxon.length];
			for(int i=0;i<taxon.length;i++){
				labelSet[i]=taxon[i].getLabel();
			}
		}
		return labelSet;
	}
	
	public static VisRDGUI getGUI(){ return GUI; } // Temporary method

	public static void setDefaultWindowSize(){
		System.out.println("Set default size:" + sequenceLength);
		System.out.println("Round:" + Math.round(0.000575 * sequenceLength));
		
		int suggestedSize = (int)Math.round(0.000575 * sequenceLength)*100;
		
		if(!windowSizeSpecified){ windowSize = suggestedSize; }
		if(!stepSizeSpecified){ stepSize = suggestedSize/5; }
	}

    static File lastOpenFile = new File (System.getProperty ("user.dir"));

	public static void main (String [] args){
		try { run (args); }
		catch (Exception e){
			System.out.println("An error occurred while starting the program: " + e.getMessage ());
			e.printStackTrace();
		}
	}

    public static void run(String [] args) throws Exception{
        java.util.Hashtable at = IOHandler.processArguments(args);
        
        if(at.get("-h")!=null){
        	// Print a help message
        	System.out.println(PROGRAMNAME + " " + VERSION);
			System.out.println("usage: java -jar visrd.jar [-c   dfhlLnNrtvV19] [-S suffix] [file ...]");
			System.out.println("-c	run on command-line only");
			System.out.println("-i	input file NESUS/FASTA");
			System.out.println("-a	auto-assign taxa to groups");
			System.out.println("-s	step size");
			System.out.println("-w	window size");
			System.out.println("-d	weighting method, integer 0..5");
			System.out.println("-bs	position in alignment to begin scan");
			System.out.println("-es	position in alignment to end scan");
			System.out.println("-n	generate a null distribution of specified number of datasets");
			System.out.println("-r	NEXUS file containing replicate datasets");
			System.out.println("Report bugs to <martin.lott@uea.ac.uk>");
        	System.exit(0);
        }
        
        
        showGUI=(at.get("-c")==null);
        
        File file = null;
        String infile = (String)at.get("-i");

//        NewNexusDocument docP = new NewNexusDocument();

        
        if (infile!=null){ file = new File (infile); }
        else{
            JFileChooser chooser = new JFileChooser (lastOpenFile);
            chooser.addChoosableFileFilter (new FastaFilter());
            chooser.addChoosableFileFilter (new NexusFilter());
                         
            if(chooser.showOpenDialog(null)==JFileChooser.APPROVE_OPTION){
                file = chooser.getSelectedFile ();
                lastOpenFile = file;
            }
        }
        
        if(file == null||file.isFile ()==false){
        	VisRDGUI.displayError("No valid input file given");
        	System.exit(0);
        }
        
        IOHandler.loadFile(file);

      	System.out.println("about to load visrd");


        ScanWindow.init(taxon);
		setSequenceLength (taxon[0].getNchar ());

        VisRDGUI.updateDatasets();
        
      	System.out.println("loaded visrd");
        
        if((String)at.get("-a")!=null){
	        ArrayList<String> groupNamesTemp = new ArrayList<String>();
	        ArrayList<ArrayList> taxaGroups = new ArrayList<ArrayList>();
	        
	    	for(int i=0;i<taxon.length;i++){
				char groupChar = taxon[i].getLabel().charAt(0);
				String thisChar = Character.toString(groupChar);
				
				if(Character.isDigit(groupChar)){
					thisChar = taxon[i].getLabel().substring(0,2);
				}
				
				int index = groupNamesTemp.indexOf(thisChar);
				
				if(index==-1){
					ArrayList<Integer> list = new ArrayList<Integer>();
					list.add(new Integer(i));
					taxaGroups.add(list);
					groupNamesTemp.add(thisChar);
				}
				else{
					ArrayList<Integer> list = taxaGroups.get(index);
					list.add(new Integer(i));
				}
	    	}
	    	
	    	groupNames = new Object[groupNamesTemp.size()];
	    	for(int i=0;i<groupNamesTemp.size();i++){
	    		groupNames[i] = groupNamesTemp.get(i);
	    	}
	    	
	        setupTaxaGroups(groupNamesTemp.size()+1);
	        
	        for(int i=0;i<taxaGroups.size();i++){
	        	ArrayList<Integer> aList = taxaGroups.get(i);
	        	int[] list = new int[aList.size()];
	        	
	        	System.out.print("Group " + groupNames[i] + ": {");
	        	
	        	for(int j=0;j<aList.size();j++){
	        		list[j] = aList.get(j).intValue();
	        		System.out.print(taxon[list[j]].getLabel() + ",");
	        	}
	        	
	        	System.out.println("}");
	        	setTaxaGroup(i+1, list);
	        }
	        
	        clusterQuartets=true;
        }
        
        String stepSizeStr = (String)at.get("-s");
        if(stepSizeStr!=null){ stepSize = Integer.parseInt(stepSizeStr); }
        stepSizeSpecified=(stepSizeStr!=null);
        
        String windowSizeStr = (String)at.get("-w");
        if(windowSizeStr!=null){ windowSize = Integer.parseInt(windowSizeStr); }
        windowSizeSpecified=(windowSizeStr!=null);

        String distanceMethod = (String)at.get("-d");
        if(distanceMethod!=null){ ScanWindow.setMethodType(Integer.parseInt(distanceMethod)); }
        
       	String startSet = (String)at.get("-bs");
        if(startSet!=null){
   	        ScanWindow.setStartDatasetCLI(Integer.parseInt(startSet));
        }
 
        String endSet = (String)at.get("-es");
        if(endSet!=null){
   	        ScanWindow.setEndDatasetCLI(Integer.parseInt(endSet));
        }
        
        VisRD.setDatasetCounter(1);
        String nullDistribution = (String)at.get("-n");
        if(nullDistribution!=null){
        	// r/s denoted shuffling method
        	// number is number of datasets to generate	
		//	VisRD.scan(shuffleChoice.getSelectedIndex(),outputToFile.isSelected(),getNdatasets());
			VisRD.scan(1,false,Integer.parseInt(nullDistribution));
        }

        
        String replicateFile = (String)at.get("-r");
        if(replicateFile!=null){
			KimuraDistanceFunction.setParameters(4.8);
        	
        	forceTaxa = new int[0];
        	includeTaxa = new int[taxon.length];
        	for(int i=0;i<taxon.length;i++){
        		includeTaxa[i]=i;
        	}

			desiredQuartets = (taxon.length) * (taxon.length - 1) * (taxon.length - 2) * (taxon.length - 3) / 24;
			
			firstWindow=windowSize/2;
        	lastWindow=taxon[0].getNchar ()-windowSize/2;

        	file = new File(replicateFile);
        	IOHandler.loadFile(file);
        	
        	VisRDGUI.updateDatasets();
        	
        	NexusIO.loadData(file); // Start processing the replicate sets...
        }
        
        
        if(at.get("-p")!=null){ // Process input file '-i' straight away...
/*        	System.out.println("processing file");
        	
			KimuraDistanceFunction.setParameters(4.8);
        	
        	forceTaxa = new int[0];
        	includeTaxa = new int[taxon.length];
        	for(int i=0;i<taxon.length;i++){
        		includeTaxa[i]=i;
        	}

			desiredQuartets = (taxon.length) * (taxon.length - 1) * (taxon.length - 2) * (taxon.length - 3) / 24;
			
			firstWindow=windowSize/2;
        	lastWindow=taxon[0].getNchar ()-windowSize/2;
        	
        	
        	VisRD aVisRD = new VisRD();
        	
			try { // running VisRD analysis
				aVisRD.start ();
			}
			catch (Exception e) {
				VisRDGUI.displayError ("Error while scanning: " + e.getMessage ());
			}
*/      }
        else{
        	if(showGUI){
        		GUI = new VisRDGUI ();
        
		        GUI.setTitle (file.getName ()); 
		        ScanWindow.init(VisRD.getAllTaxon());
		        GUI.updateDataset ();
        	}
        }
    }
}