package visrd;

/**

   This file is part of VisRD

   Copyright (C) 2002 Kristoffer Forslund (jeanpaulsartre@hotmail.com)
   for the Linnaeus Centre for Bioinformatics, Uppsala University

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with VisRD; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/**

	Superclass for classes that take characters and quartets and return trajectories...

*/

import java.util.Random;

public class TrajectoryCalculator {
	// The sequence data
	protected Taxon[] taxon;
	private boolean warning;

	protected int firstWindow;
	protected int lastWindow;
	protected int stepSize;
	protected int windowSize;
	
	protected boolean[] mask;
	protected boolean useMask;
	
	private int index=0;
	private Triplet[] result;
	
	private static double minimumAlpha;

	// Constructor - loads with the characters
	public TrajectoryCalculator (Taxon[] taxon){
		this.taxon = taxon;
		warning = false;
		Taxon.setIsDNAorRNA();
		minimumAlpha = Double.MAX_VALUE;
	}

	// Calculate all trajectories for a given Quartet
	public Triplet[] calculate (Quartet q){
		boolean persistTrajectories = true;//(VisRD.getDisplayQuartets()!=0);
		// Calculate all the trajectories of a particular quartet
		// This is where I need to add that rotation thingy

		warning = false;
				
		/**
		
			Loop along the sequence, adding triplets to result
		
			This must basically: loop from offset + windowSize / 2 to end - windowSize / 2 or so
			by stepSize increments, taking out a windowSize-sized window for the four
			sequences, checking for column patterns via method-specific algorithm,
			keeping score, calculating triplets, adding them to result
		
			The sequences can be acquired by Characters.getRow (i), then there is the
			masking info
		
		*/
	
		/**
		
			Assume we have start and end centre positions known...
		
		*/
	
		firstWindow = VisRD.getFirstWindow ();
		lastWindow = VisRD.getLastWindow ();
		stepSize = VisRD.getStepSize ();
		windowSize = VisRD.getWindowSize ();
	
		mask = null; //chars.getMask ();

		if(mask == null){ useMask = false; }
		else{ useMask = true; }
	
		// Call a warm up on first window to allow rotation as nessesary
		/* Swap this to middle lane if not already there therefore
		* modify the quartet swapping indexes to reflect this */

		Triplet tempT = processWindow(firstWindow,
										taxon[q.getX()].getMatrix(),
										taxon[q.getY()].getMatrix(),
										taxon[q.getU()].getMatrix(),
										taxon[q.getV()].getMatrix(),q);
		swapToMiddleLane(tempT.toPolar(), q);
	
		// The chars themselves...
		char[] rowX = taxon[q.getX()].getMatrix(); // The sequence for taxa pointed to by 'q.getX()'
		char[] rowY = taxon[q.getY()].getMatrix();
		char[] rowU = taxon[q.getU()].getMatrix();
		char[] rowV = taxon[q.getV()].getMatrix();

		int size = 1 + (int)((double)(lastWindow-firstWindow)/(double)stepSize);
	
		if(persistTrajectories){
			result = new Triplet[size];
		
			int i=0;
			
			double minA=Double.MAX_VALUE,maxA=Double.MIN_VALUE,minB=Double.MAX_VALUE,maxB=Double.MIN_VALUE;
			
			for (int s = firstWindow; s <= lastWindow; s += stepSize) {
				Triplet t = processWindow(s,rowX,rowY,rowU,rowV,q);				
				result[i++]=t;
			}

			return result;
		}
		else{
			int distanceMetric = VisRD.getVarianceType();
			
			// Compute the mean position of the quartet
			double meanX = 0.0;
			double meanY = 0.0;
			int nTrajectories=0;
			
			for (int s = firstWindow; s <= lastWindow; s += stepSize) {
				Triplet t = processWindow(s,rowX,rowY,rowU,rowV,q);
				
				meanX += t.getA();
				meanY += t.getB();
				nTrajectories++;
			}

			meanX = meanX/nTrajectories;
			meanY = meanY/nTrajectories;
			q.setMean(meanX, meanY);
			
			Triplet meanT = new Triplet (meanX, meanY, 0).toPolar (); // May be needed
		
			// Compute max distance to the mean position
			double maxDist = 0.0;
			int i=0;
			
			for (int s = firstWindow; s <= lastWindow; s += stepSize) {
				Triplet p = processWindow(s,rowX,rowY,rowU,rowV,q);
				
				double dist =0.0;
				switch(distanceMetric){
					case Quartet.EUCLIDEAN:
						dist = Math.sqrt ((p.getA () - meanX) * (p.getA () - meanX) + (p.getB () - meanY) * (p.getB () - meanY));
					break;
					case Quartet.PATH:
						dist = Math.abs (p.getA () - meanT.getA ()) * (p.getB ());
					break;
				}
				
				if(dist > maxDist){
					maxDist = dist;
					q.setMaxPosition(i);
				}
				
				i++;
			}

			q.setVariance(maxDist);
			
			return null;
		}
	}
	
	public Triplet processWindow(int s, char[] rowX, char[] rowY, char[] rowU, char[] rowV, Quartet q){
		VisRDGUI.displayError("processWindow() must be overridden in the subclass");	
		return null;
	}
	
	public Triplet generateTriplet(int s, double s1, double s2, double s3){
		// We are given weightings for the three different scenarios
		// s1 = XY|UV, s2 = XU|YV, s3 = XV|YU
		
		if (s1 + s2 + s3 != 0) {
			double alphaA = (-1.0/3.0)/(s1 - (1.0/3.0));
			double alphaB = (-1.0/3.0)/(s2 - (1.0/3.0));
			double alphaC = (-1.0/3.0)/(s3 - (1.0/3.0));

			
			double alphaWeWant = Double.MAX_VALUE;
			if(alphaA>0&&alphaA<alphaWeWant){ alphaWeWant = alphaA; }
			if(alphaB>0&&alphaB<alphaWeWant){ alphaWeWant = alphaB; }
			if(alphaC>0&&alphaC<alphaWeWant){ alphaWeWant = alphaC; }				

			if(alphaWeWant<minimumAlpha){
				minimumAlpha = alphaWeWant;
//				System.out.println("alphaA:"+ alphaA + "alphaB:"+ alphaB + "alphaC:"+ alphaC + "alphaWeWant:" + alphaWeWant);
//				System.out.println("minimumAlpha:" + minimumAlpha);
			}
			
			// X is support for s3 - s1
			double x = ((s3 / (s1 + s2 + s3)) - (s1 / (s1 + s2 + s3))) / Math.sqrt (2.0);
			
			// Y is support for s2
			double y = ((s2 / (s1 + s2 + s3)) - (1.0 / 3.0)) * Math.sqrt (3.0 / 2.0);

			Triplet t = new Triplet (x, y, s);
			t.setOriginalTopologySupport(new Triplet(s1,s2,s3));
			return t;
		}
	
		else{ //No information...
			warning = true;
			return new Triplet (0, 0, s);
		}
	}
	
	public void swapToMiddleLane(Triplet polarTriplet, Quartet q){
		/* Analyse the angle of the polar triplet and if it's not
		 * in the middle lane then update q to ensure it will start there
		 * NOTE: s1 = XY|UV, s2 = XU|YV, s3 = XV|YU */
		double phi = polarTriplet.getA();
		double oneThird = 1.0/3.0;
		if(phi<-oneThird){
			// Bottom right triangle, scenarios 1 and 2 cause this so flip u & y
// Old method flips Y axis...
//			int temp = q.getU();
//			q.setU(q.getY());
//			q.setY(temp);

			// New Method
			int oldU = q.getU();
			int oldV = q.getV();
			q.setU(q.getY());
			q.setV(oldU);
			q.setY(oldV);
		}
		else if(phi>oneThird){
			// Bottom left triangle, scenarios 2 and 3 cause this so flip u & v
// Old method flips Y axis...
//			int temp = q.getU();
//			q.setU(q.getV());
//			q.setV(temp);
			
			// New Method
			int oldU = q.getU();
			int oldV = q.getV();
			q.setU(oldV);
			q.setV(q.getY());
			q.setY(oldU);
		}
	}

	public boolean getWarning (){ return warning; }
	public static double getMinimumAlpha(){ return minimumAlpha; }
}
