/* This is the Quartet selction panel */

package visrd;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.awt.*;
import java.awt.event.*;
import java.util.ArrayList;
import java.util.Vector;

public class TaxonSelectorPanel extends JPanel implements ActionListener, ChangeListener{	
	private String[] names;
	private Taxon[] taxon;
	
	private JPanel top, groups, main;
	private JScrollPane centerScrollPane;

	private JCheckBox useClusterModel, outputToFile;
	
	// Non-cluster model
	private JList forceList, includeList, excList;
	private DefaultListModel forceModel, includeModel, excModel, groupNameModel;
	private JButton incInAllToInc, incToIncInAll, incToExc, excToInc; // Buttons to move items between lists
	
	// Used for the cluster model
	private JComboBox[] clusterSelection;
	
	private JList groupList;
	private JTextField groupName;
	private JButton addGroup, updateGroup, removeGroup, defaultGroups, autoGroups;
	
	public TaxonSelectorPanel (){
		super(new BorderLayout());
	}
	
	public void init(Taxon[] taxon){
		this.taxon=taxon;
		top=null;
		centerScrollPane=null;
		groups=null;
		main=null;
		clusterSelection=null;
		
		updateDisplay();
	}

	public void updateDisplay(){
		// Set taxon names
		names = new String[taxon.length];
		for(int i = 0;i<taxon.length;i++) {
			names[i] = taxon[i].getLabel();
		}

		// Do top panel
		if(top==null){
			top = new JPanel();
			
			useClusterModel = new JCheckBox("Use group model");
			useClusterModel.addChangeListener(this);
			top.add(useClusterModel);
			
			outputToFile = new JCheckBox("Output quartets to file");
			outputToFile.addChangeListener(this);
			top.add(outputToFile);
		}

		drawLowerPanel();
	}
	
	private void drawLowerPanel(){
		removeAll();
		add(top,BorderLayout.NORTH);

		if(useClusterModel.isSelected()){ drawClusterSelection(); }
		else{ drawLists(); }
	}

	private void drawClusterSelection(){
		// Draws the cluster selection model
		// This is the list of taxa and their groups
		if(centerScrollPane==null){
			JPanel center = new JPanel();
			center.setBorder(BorderFactory.createTitledBorder("Assign taxon to groups (clusters)"));
			center.setLayout(new GridLayout(names.length,2));
			
			setDefaultGroupNames();
			setGroupSelection();
			
			for(int i=0;i<names.length;i++){
				center.add(new JLabel(names[i]));
				center.add(clusterSelection[i]);
			}
			
			centerScrollPane=new JScrollPane(center);
			centerScrollPane.setPreferredSize(new Dimension(450, 110));
		}
		add(centerScrollPane,BorderLayout.CENTER);
			
		if(groups==null){
			// This is the list of groups
			groups = new JPanel();
			groups.setBorder(BorderFactory.createTitledBorder("Available groups"));
			groups.setLayout(new BorderLayout());
			
			// Use a grid bag
			JPanel north = new JPanel(new GridBagLayout());
			
        	GridBagConstraints c = new GridBagConstraints();
			c.fill = GridBagConstraints.HORIZONTAL;

			groupName = new JTextField();
			c.gridx = 0;
			c.gridwidth = 6;
			c.gridy = 0;
			north.add(groupName, c);
			
			
			c.gridy = 1;
			c.gridwidth = 2;
			addGroup = new JButton("Add");
			addGroup.addActionListener(this);
			north.add(addGroup, c);
			
			c.gridx = 2;
			updateGroup = new JButton("Update");
			updateGroup.addActionListener(this);
			north.add(updateGroup, c);

			c.gridx = 4;			
			removeGroup = new JButton("Remove");
			removeGroup.addActionListener(this);
			north.add(removeGroup, c);
			
			// 	Start second row of buttons
			c.gridy = 2;
			c.gridx = 0;
			c.gridwidth = 3;
			defaultGroups = new JButton("Reset");
			defaultGroups.addActionListener(this);
			north.add(defaultGroups, c);

			c.gridx = 3;
			autoGroups = new JButton("Auto Assign");
			autoGroups.addActionListener(this);
			north.add(autoGroups, c);
			
			groups.add(north,BorderLayout.NORTH);	

			// Start the JList stuff
			setDefaultGroupNames();
			groupList = new JList(groupNameModel);
			groupList.addMouseListener(new MouseAdapter(){
				public void mouseClicked(MouseEvent e){
					groupName.setText((String)groupList.getSelectedValue());
				}
			});
			
			groups.add(groupList,BorderLayout.CENTER);
		}
		add(groups,BorderLayout.EAST);
	}
	
	private void drawLists(){
		if(main==null){
			// Draws three lists
			main = new JPanel();
			main.setLayout(new GridLayout(1,3));
			// A box of BorderLayouts
	
			String[] listNames = {"Included in all quartets","Included","Not included"};

			for(int i=0;i<listNames.length;i++){
				JPanel listPanel = new JPanel(new BorderLayout());
				listPanel.setBorder(BorderFactory.createTitledBorder(listNames[i]));
				JScrollPane	listScrollPane=null;
	
				switch(i){ // Setup the three different lists
					case 0:
						forceModel = new DefaultListModel();
						forceList = new JList(forceModel);
						
						listScrollPane=new JScrollPane(forceList);
						listScrollPane.setPreferredSize(new Dimension(100, 250));
				
						listPanel.add(listScrollPane,BorderLayout.CENTER);
						incInAllToInc = new JButton(">>");
						incInAllToInc.addActionListener(this);
						listPanel.add(incInAllToInc,BorderLayout.SOUTH);
					break;
					case 1:	
						includeModel=new DefaultListModel();
						
						for(int n=0;n<names.length;n++){
							includeModel.addElement(names[n]);
						}
						includeList = new JList(includeModel);
						
						listScrollPane=new JScrollPane(includeList);
						listScrollPane.setPreferredSize(new Dimension(100, 250));
						
						listPanel.add(listScrollPane,BorderLayout.CENTER);
						
						JPanel buttonPanel = new JPanel();
						buttonPanel.setLayout(new GridLayout(1,2));
						
						incToIncInAll = new JButton("<<");
						incToIncInAll.addActionListener(this);
						buttonPanel.add(incToIncInAll);
						
						incToExc = new JButton(">>");
						incToExc.addActionListener(this);
						buttonPanel.add(incToExc);
	
						listPanel.add(buttonPanel,BorderLayout.SOUTH);
					break;
					case 2:
						excModel=new DefaultListModel();
						excList = new JList(excModel);
						
						listScrollPane=new JScrollPane(excList);
						listScrollPane.setPreferredSize(new Dimension(100, 250));
						
						listPanel.add(listScrollPane,BorderLayout.CENTER);
						excToInc = new JButton("<<");
						excToInc.addActionListener(this);
						listPanel.add(excToInc,BorderLayout.SOUTH);
					break;
				}

				main.add(listPanel);
			}
		}
		add(main,BorderLayout.CENTER);
	}
	
	private void setDefaultGroupNames(){
		if(groupNameModel==null){
			groupNameModel = new DefaultListModel();
		}
		
		if(groupNameModel==null||groupNameModel.isEmpty()){
			for(int i=1;i<5;i++){
				groupNameModel.addElement("Group " + i);
			}	
		}
	}

	private void setGroupSelection(){ setGroupSelection(false); }
	
	private void setGroupSelection(boolean byIndex){
		if(clusterSelection==null){
			clusterSelection = new JComboBox[names.length];
			Object[] groupNames = groupNameModel.toArray();
			VisRD.setGroupNames(groupNames);
			
			for(int i=0;i<names.length;i++){
				clusterSelection[i] = new JComboBox();
				clusterSelection[i].addItem("No group");
				
				for(int j=0;j<groupNames.length;j++){
					clusterSelection[i].addItem(groupNames[j]);
				}
			}
		}
		else{
			Object[] selectedItems = null;
			int[] selectedIndexes = null;
			
			if(byIndex){ // Remember the indexes
				selectedIndexes = new int[clusterSelection.length];
				for(int i=0;i<names.length;i++){
					selectedIndexes[i] = clusterSelection[i].getSelectedIndex();
				}
			}
			else{ // Remember thhe items
				selectedItems = new Object[clusterSelection.length];
				for(int i=0;i<names.length;i++){
					selectedItems[i] = clusterSelection[i].getSelectedItem();
				}
			}
			


			Object[] groupNames = groupNameModel.toArray();
			VisRD.setGroupNames(groupNames);
			
			for(int i=0;i<names.length;i++){
				clusterSelection[i].removeAllItems();
				clusterSelection[i].addItem("No group");

				for(int j=0;j<groupNames.length;j++){
					clusterSelection[i].addItem(groupNames[j]);
				}
			}
			
			
			if(byIndex){ // Remember the indexes
				for(int i=0;i<names.length;i++){
					clusterSelection[i].setSelectedIndex(selectedIndexes[i]);
				}
			}
			else{
				for(int i=0;i<names.length;i++){
					clusterSelection[i].setSelectedItem(selectedItems[i]);
				}
			}
		}
	}
	
	public void actionPerformed (ActionEvent e){
		if(e.getSource()==incInAllToInc){
			transferSelectedItem(forceList,forceModel,includeModel);
		}
		else if(e.getSource()==incToIncInAll){
			transferSelectedItem(includeList,includeModel,forceModel);
		}
		else if(e.getSource()==incToExc){
			transferSelectedItem(includeList,includeModel,excModel);
		}
		else if(e.getSource()==excToInc){
			transferSelectedItem(excList,excModel,includeModel);
		}
		else if(e.getSource()==addGroup){
			groupNameModel.addElement(groupName.getText());
			setGroupSelection();
			drawLowerPanel(); // Only done so color quartet selection is updated
		}
		else if(e.getSource()==updateGroup){
			groupNameModel.setElementAt(groupName.getText(),groupList.getSelectedIndex());
			setGroupSelection(true);
			drawLowerPanel();
		}
		else if(e.getSource()==removeGroup){
			groupNameModel.remove(groupList.getSelectedIndex());
			setGroupSelection();
			drawLowerPanel();
		}
		else if(e.getSource()==defaultGroups){
			groupNameModel.removeAllElements();
			setDefaultGroupNames();
			
			setGroupSelection();
			drawLowerPanel();
		}
		else if(e.getSource()==autoGroups){
			groupNameModel.removeAllElements();
			char[] prevElement = new char[taxon.length];

			for(int i=0;i<taxon.length;i++){
				char groupChar = taxon[i].getLabel().charAt(0);
				String thisChar = Character.toString(groupChar);
				
				if(Character.isDigit(groupChar)){
					thisChar = taxon[i].getLabel().substring(0,2);
				}
				
				int index=-1;
				for(int j=0;j<groupNameModel.size();j++){
					if(thisChar.equals(groupNameModel.getElementAt(j))){
						index=j;
					}
				}
				
				
				if(index==-1){
					groupNameModel.addElement(thisChar);
					setGroupSelection();
					index=groupNameModel.size()-1;
				}
				
				clusterSelection[i].setSelectedIndex(index+1);
			}
			drawLowerPanel();
		}
	}
	
	private void transferSelectedItem(JList list1, DefaultListModel model1, DefaultListModel model2){
		Object obj = list1.getSelectedValue();
		int index = list1.getSelectedIndex();
		
		if(obj!=null){
			model1.removeElement(obj);
			model2.addElement(obj);	
		}
		
		if(index==model1.getSize()){
			list1.setSelectedIndex(index-1);
		}
		else{
			list1.setSelectedIndex(index);
		}
	}
	
	public void stateChanged(ChangeEvent e){
		if(e.getSource()==outputToFile){
			VisRD.setOutputQuartets(outputToFile.isSelected());
		}
		else{
			drawLowerPanel();
			repaint();
		}
	}

	private void harvestClusterLists(){
		// Much more efficient implementation of harvesting the cluster lists
		
		//To extend this we first need to work out the maximum cluster number the user wanted
		
		int maxGroup = 0;
		for (int n = 0; n < clusterSelection.length; n++) {
			JComboBox cC = clusterSelection[n];
			int group=cC.getSelectedIndex();
			if(group>maxGroup){ maxGroup=group; }
		}

		VisRD.setupTaxaGroups(maxGroup+1);
		int[] count = new int[maxGroup+1];
		for (int n = 0; n < clusterSelection.length; n++) {
			JComboBox cC = clusterSelection[n];
			int group=cC.getSelectedIndex();
			if(group!=0){ count[group]++; }
		}
		
		// Check if at least 4 groups in use...
		int numberOfGroupsInUse=0;
		
		int[][] result=new int[count.length][];
		for(int i=1;i<count.length;i++){
			result[i] = new int[count[i]];
			if(count[i]!=0){ numberOfGroupsInUse++; }
			count[i]=0;
		}
		
		if(numberOfGroupsInUse<4){
			VisRDGUI.displayError("There are less than four groups in use so VisRD is unable to generate quartets");
		}
		
		for (int n = 0; n < clusterSelection.length; n++) {
			JComboBox cC = clusterSelection[n];
			int group=cC.getSelectedIndex();
			if(group!=0){ result[group][count[group]++] = n; }
		}

		for(int i=1;i<count.length;i++){
			VisRD.setTaxaGroup (i, result[i]);
		}
	}
	
	public void harvest(){ // Also need to harvest which taxa are flagged
		if(useClusterModel.isSelected()){
			// For cluster model
			harvestClusterLists();
		}
		else{
			// For non-cluster model
			harvestIncludeTaxa();
			harvestForceTaxa();
		}
	}

	public void harvestIncludeTaxa(){
		ListModel lm = includeList.getModel();
		int[] includeTaxa = new int[lm.getSize()];
	
		for (int i = 0;i<lm.getSize();i++) {
			String name = (String)lm.getElementAt(i);
			for(int j=0;j<names.length;j++){
				if(names[j].equals(name)){
					includeTaxa[i] = j;
				}
			}
		}
		
		VisRD.setIncludeTaxa (includeTaxa);
	}
	
	public void harvestForceTaxa (){
		ListModel lm = forceList.getModel();
		int[] forceTaxa = new int[lm.getSize()];
	
		for (int i = 0;i<lm.getSize();i++) {
			String name = (String)lm.getElementAt(i);
			for(int j=0;j<names.length;j++){
				if(names[j].equals(name)){
					forceTaxa[i] = j;
				}
			}
		}
		
		VisRD.setForceTaxa(forceTaxa);
	}
	
	public void harvestParameters(){
		VisRD.setClusterQuartets (useClusterModel.isSelected());
	}
}