package visrd;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.*;
import java.io.*;


/**
	Taxon Ranking panel
*/

public class TaxonRankingPanel extends JPanel implements ActionListener{
	private Taxon[] taxon;
	private QuartetSet quartets;
	private static double average, variance, coVariance;
	
	private JLabel summary;
	private JComboBox meanType;
	private JList taxa,frequencies;
	
	private static RankingResults rankingResults;

	public TaxonRankingPanel(QuartetSet quartets, Taxon[] taxon){
		super(new BorderLayout());
		
		summary = new JLabel();
		add(summary, BorderLayout.NORTH);
		
		
		JSplitPane jsp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);

		this.taxon=taxon;
		this.quartets=quartets;

		taxa = new JList();
		frequencies = new JList();
		
		MouseListener mouseListener = new MouseAdapter() {
		     public void mouseReleased(MouseEvent e) {
				if(e.getSource()==taxa){
					frequencies.setSelectedIndex(taxa.getSelectedIndex());
				}
				else{
					taxa.setSelectedIndex(frequencies.getSelectedIndex());
				}
		     }
 		};
		taxa.addMouseListener(mouseListener);
		frequencies.addMouseListener(mouseListener);

        jsp.setResizeWeight (0.5);


		JPanel leftPane = new JPanel(new BorderLayout());
		
		String name=null;
		if(VisRD.getClusterQuartets()){ name="Group name"; }
		else{ name="Taxon name"; }

		JLabel taxonName = new JLabel(name);
		taxonName.setPreferredSize(new Dimension(100,25));
		leftPane.add(taxonName, BorderLayout.NORTH);
		
		leftPane.add(new JScrollPane(taxa), BorderLayout.CENTER);
		jsp.setLeftComponent(leftPane);
	
	
		JPanel rightPane = new JPanel(new BorderLayout());
		
		String[] options = {"Mean distance", "Mean index"};
		meanType = new JComboBox(options);
		meanType.addActionListener(this);
		
		rightPane.add(meanType, BorderLayout.NORTH);
		rightPane.add(new JScrollPane(frequencies), BorderLayout.CENTER);
		jsp.setRightComponent(rightPane);

		redrawLists();
		
		add(jsp, BorderLayout.CENTER);
	}
	
	public TaxonRankingPanel(QuartetSet quartets, Taxon[] taxon, boolean meanDistance){
		getRankingResults(quartets,taxon,meanDistance);
	}
	
	public static RankingResults getRankingResults(QuartetSet quartets, Taxon[] taxon, boolean meanDistance){
		Rank[] data = quartets.frequencies (taxon.length,meanDistance);

		int totalIncludedTaxa=data.length;
		if(!VisRD.getClusterQuartets()){
			totalIncludedTaxa = VisRD.getForceTaxa().length + VisRD.getIncludeTaxa().length;
		}

		computeCV(data, totalIncludedTaxa);
		

		
		if(VisRD.getClusterQuartets()){
			int numberOfGroups = VisRD.getTaxaMatrix().length;
			int[][] taxaMatrix = VisRD.getTaxaMatrix();
			
			rankingResults = new RankingResults(numberOfGroups+1);
			rankingResults.setValues(0,average,variance,coVariance);
			
			// Test removing each group
			for(int i=1;i<numberOfGroups;i++){
				Rank[] maskedData = quartets.frequencies (taxon.length,meanDistance,taxaMatrix[i]);
				computeCV(maskedData, totalIncludedTaxa-1);
				
			//	System.out.println("coVariance " + i + ": " + coVariance);
				rankingResults.setValues(i,average,variance,coVariance);
			}
		}
		else{
			rankingResults = new RankingResults(taxon.length+1);
			rankingResults.setValues(0,average,variance,coVariance);
			
			for(int i=0;i<taxon.length;i++){
				int[] index = new int[1];
				index[0] = i;
				
				Rank[] maskedData = quartets.frequencies (taxon.length,meanDistance,index);
				computeCV(maskedData, totalIncludedTaxa-1);
				
			//	System.out.println("coVariance " + i + ": " + coVariance);
				rankingResults.setValues(i+1,average,variance,coVariance);
			}
		}

		return rankingResults;
	}
	
	public void actionPerformed(ActionEvent e){
		redrawLists();
	}
	
	private void redrawLists(){
		Rank[] data = quartets.frequencies (taxon.length,meanType.getSelectedIndex()==1);
		
		// Output hashtables
/*		System.out.println("Output hashtables");
		try{
			java.io.PrintWriter pw = new java.io.PrintWriter(new java.io.BufferedWriter(new java.io.FileWriter("histograms.csv")));
			for(int i=data.length-1;i>=0;i--){
				pw.println(data[i].hashtableToString());
			}
			pw.flush();
			pw.close();
		}
		catch(Exception e){ e.printStackTrace(); }
*/		
		
		int totalIncludedTaxa=data.length;
		
		if(!VisRD.getClusterQuartets()){
			totalIncludedTaxa = VisRD.getForceTaxa().length + VisRD.getIncludeTaxa().length;
		}
		
		String[] names = new String[totalIncludedTaxa];
		String[] frequencyData = new String[totalIncludedTaxa];
		int index=0;
		double maximumRank = -1;
		
		for(int i=0;i<data.length;i++){
			// Maximum rank is first real result
			if(data[i].getQuartetCount()!=0){
				if(data[i].getValue()>maximumRank){
					maximumRank=data[i].getValue();
				}
			}
		}
		
		for(int i=0;i<data.length;i++){
			Rank rank = data[data.length-i-1];
			int quartetCount = rank.getQuartetCount();
			if(quartetCount!=0){
				names[index]=rank.getLabel();
//				int percentage = (int)Math.round((maximumRank - rank.getValue())*10000/maximumRank);
				int percentage = (int)Math.round(rank.getValue()*10000/maximumRank);
//				System.out.println(rank.getValue() + " out of " + maximumRank);
				frequencyData[index]=Double.toString(rank.getValue()) + " based on " + quartetCount + " quartets. " + (percentage/100) + "." + makeTwoDigit(percentage%100) + "% of the maximum.";
				index++;
			}
		}

		taxa.setListData(names);
		frequencies.setListData(frequencyData);
		
		computeCV(data, totalIncludedTaxa);
		updateSummary(average, variance, coVariance);
/*		
		if(VisRD.getClusterQuartets()){
			// Test removing each group
		}
		else{
			// Play around with difference CVs
			for(int i=0;i<taxon.length;i++){
				int[] ind = new int[1];
				ind[0] = i;
				Rank[] maskedData = quartets.frequencies (taxon.length,meanType.getSelectedIndex()==1,ind);
				computeCV(maskedData, totalIncludedTaxa-1);
				System.out.println("CV for " + taxon[i].getLabel() + " is " + coVariance);
			}
		}
*/	}
	
	private static void computeCV(Rank[] data, int totalIncludedTaxa){
		// Set up the variance and coefficient of variance
		double total=0.0;
		for(int i=0;i<data.length;i++){
			if(data[i].getQuartetCount()!=0){ total+=data[i].getValue(); }
		}
		
		average = total/(double)totalIncludedTaxa;
		
		// Work out the variance
		variance=0.0;
		
		for(int i=0;i<data.length;i++){
			if(data[i].getQuartetCount()!=0){
				double difference = data[i].getValue()-average;
				variance+=difference*difference;
			}
		}
		
		variance = variance/(double)(totalIncludedTaxa-1);
		
		// Work out the coeffiecient of variance
		double sd_dev = Math.sqrt(variance);
		coVariance = sd_dev/average;
	}
	
	private void updateSummary(double average, double variance, double cv){
		String avgStr = Double.toString(average);
		String varStr = Double.toString(variance);
		String cvStr = Double.toString(coVariance);

		int nDatasets=0;
		int nDatasetsLarger=0;

		String pathPlusFile = VisRD.getCurrentPathFilename();
		int endIndex = pathPlusFile.length()-1;
		int index=endIndex;
		
		while(endIndex==pathPlusFile.length()-1){
			if(pathPlusFile.charAt(index)==File.separatorChar){
				endIndex=index;
			}
			else{ index--;}
		}
		
		try{
			String currentFile = pathPlusFile.substring(endIndex+1,pathPlusFile.length());
			File dir = new File(pathPlusFile.substring(0,endIndex));
			File[] files = dir.listFiles();
			
			for(File f:files){
				String name = f.getName();
				if(name.startsWith(currentFile)){
					if(name.substring(name.length()-3,name.length()).equalsIgnoreCase("csv")){
						BufferedReader br = new BufferedReader(new FileReader(f));
						br.readLine(); // Skip the first line
						
						String averages = br.readLine();
						String[] data = averages.split(",");
						nDatasets+=data.length-1;
						
						for(int i=1;i<data.length;i++){
							double otherAverage = Double.parseDouble(data[i]);
							if(otherAverage>average){ nDatasetsLarger++; }
						}
						
						br.close();
					}
				}
			}
		}
		catch(Exception e){ e.printStackTrace(); }
		
		
//		summary.setText("Average: " + avgStr + ",  Variance:" + varStr + ",  Coefficient of Variance:" + cvStr);
		double pValue = ((double)nDatasetsLarger/(double)nDatasets);
		String recStr = "No null distribution found.";
		if(nDatasets!=0){ recStr = "No significant evidence for recombination"; }
		if(pValue<=0.01){ recStr = "Evidence for recombination!"; }
		
		if(meanType.getSelectedIndex()==0){
			summary.setText("Average: " + avgStr + ",  P-value:" + pValue + " from " + nDatasets + " datasets. " + recStr);
		}
		else{
			summary.setText("Average: " + avgStr);
		}
		
	}
	
	public static RankingResults getRankingResults(){
		// Ensure the results are not sorted - otherwise the index will be corrupted!
		return rankingResults;
	}
	
	private static String makeTwoDigit(int i){
		if(i<10){ return "0" + i; }
		else{ return Integer.toString(i); }
	}
}

class Rank implements Comparable<Rank>{
	private String label; // Position of the taxa in the list
	private double value;
	private int quartetCount; // The number of quartets this taxa is in
	private Hashtable<Integer, Integer> ht;
/*	
	public Rank (String label, double value, int maxPosition){
		this(label, value, -1, maxPosition);
	}
*/	
	public Rank (String label, double value, int quartetCount, Hashtable<Integer, Integer> ht){
		this.label = label;
		this.value = value;
		this.quartetCount=quartetCount;
		this.ht = ht;
	}
	
	public String getLabel(){ return label; }
	public double getValue(){ return value; }
	public int getQuartetCount(){ return quartetCount; }
	
	public String hashtableToString(){
		StringBuilder sb = new StringBuilder();
		sb.append(label + "\n");
		sb.append("Trajectoy,Start window - End window,Frequency\n");
		
		int endIndex = (VisRD.getSequenceLength()-VisRD.getWindowSize())/VisRD.getStepSize();
		
		for(int i=0;i<endIndex;i++){
			Integer freq = ht.get(new Integer(i));
			if(freq==null){ freq = new Integer(-1); } // So the result is 0
			
			sb.append(i + "," + (i*VisRD.getStepSize()) + "-" + (i*VisRD.getStepSize()+VisRD.getWindowSize()) + " to " + ((i+1)*VisRD.getStepSize()) + "-" + ((i+1)*VisRD.getStepSize()+VisRD.getWindowSize())  + "," + (freq.intValue()+1));
			
			sb.append('\n');
		}

		return sb.toString();
	}

	public int compareTo(Rank otherVPoint){
		return Double.compare(value,otherVPoint.getValue());
	}
}

class RankingResults{
	// index 0 is average for entire set, index i is average for subset excluding taxa i-1;
	private double[] average, variance, coVariance;
	
	public RankingResults(int length){
		average = new double[length];
		variance = new double[length];
		coVariance = new double[length];
	}
	
	public void setValues(int index, double average, double variance, double coVariance){
		this.average[index]=average;
		this.variance[index]=variance;
		this.coVariance[index]=coVariance;
	}
	
	public double getAverage(int index){ return average[index]; }
	public double getVariance(int index){ return variance[index]; }
	public double getCoVariance(int index){ return coVariance[index]; }
		
	public String toString(){
		// This method will be used for debugging only	
		return "average:" + average[0] + ", variance:" + variance[0] + ", coVariance:" + coVariance[0];
	}
}