package visrd;

import java.util.ArrayList;
import java.util.ListIterator;

/**

   This file is part of VisRD

   Copyright (C) 2002 Kristoffer Forslund (jeanpaulsartre@hotmail.com)
   for the Linnaeus Centre for Bioinformatics, Uppsala University

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with VisRD; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

/**

   Class to represent a quartet, i.e. four indices to
   represent a subset of a distance matrix / set of taxa

   It is assumed that all quartet components are distinct

*/

public class Quartet implements Comparable<Quartet>{
    private int x, y, u, v;
    
    private int maxPosition;
    private double variance, meanX, meanY;
    private Triplet[] trajectories;
      
	public static final int EUCLIDEAN=0, PATH=1, ANGULAR=2;
    /*
    public Quartet () // Default constructor
    {

		// Defaults to 0. This indexes nothing

		x = 0;
		y = 0;
		u = 0;
		v = 0;
    }
*/
    /**
       Copy constructor. Note this is deep copy.
    */

    public Quartet (Quartet other){
		this(other.getX(),other.getY(),other.getU(),other.getV());
    }

    /**

       Constructor with arguments

    */

    public Quartet (int newX, int newY, int newU, int newV){
		x = newX;
		y = newY;
		u = newU;
		v = newV;
    }

    
    public int getX(){ return x; } // X accessor
    public int getY(){ return y; } // Y accessor
    public int getU(){ return u; } // U accessor
    public int getV(){ return v;} // V accessor
    public double getVariance(){ return variance; }
    public int getMaxPosition(){ return maxPosition; }
    public double getMeanX(){ return meanX; }
    public double getMeanY(){ return meanY; }
    public Triplet[] getTrajectoryList(){ return trajectories; }
    public Triplet getTrajectory(int i){ return trajectories[i]; }

    public void setX(int x){ this.x = x; } // X mutator
    public void setY(int y){ this.y = y; } // Y mutator
   	public void setU(int u){ this.u = u; } // U mutator
    public void setV(int v){ this.v = v; } // V mutator
    public void setVariance(double variance){ this.variance=variance; }
    public void setMaxPosition(int maxPosition){ this.maxPosition=maxPosition; } 
    public void setMean(double meanX, double meanY){ this.meanX=meanX; this.meanY=meanY; }
    public void setTrajectoryList(Triplet[] trajectories){ this.trajectories=trajectories; }
    
    public int compareTo(Quartet otherQuartet){ return Double.compare(variance,otherQuartet.getVariance()); }
    
    // Determines whether or not this quartet contains a specified taxa
    public boolean contains(int m){
		return (x == m ||y == m ||u == m ||v == m);
    }

    /**

       Determines whether or not a quartet contains a specified
       set of taxa

    */
/*    public boolean contains (ArrayList t){
		for (int a = 0; a < t.size (); a++) {
			if (! contains (((Integer) t.get (a)).intValue ())) {
				return false;
			}
		}

		return true;
    }
*/
    /**

       Determines whether or not a quartet is contained within
       a set of taxa, i.e. if the indices are among those
       specified

    */
/*
    public boolean isContainedIn (ArrayList t){
		return t.contains(new Integer(x))&&
				t.contains(new Integer(y))&&
				t.contains(new Integer(u))&&
				t.contains(new Integer(v));
    }
*/
    public static Quartet sizeOrder (int x, int y, int u, int v){
		// Size order is no longer needed as quartets are now swapped to the middle lane
/*		int m;
		
		if (y > x) {
			m = x;
			x = y;
			y = m;
		}

		if (u > x) {
			m = x;
			x = u;
			u = m;
		}

		if (v > x) {
			m = x;
			x = v;
			v = m;
		}

		if (u > y) {
			m = y;
			y = u;
			u = m;
		}

		if (v > y) {
			m = y;
			y = v;
			v = m;
		}

		if (v > u) {
			m = u;
			u = v;
			v = m;
		}
*/
		return new Quartet (x, y, u, v);
    }

	public boolean isUnique(){
		return (x != y && x != u && x != v && y != u && y != v && u != v);
	}
	
	public void calculateVariance(){ calculateVariance(trajectories); }
	public void calculateVariance(Triplet[] trajectories){
		int distanceMetric = VisRD.getVarianceType();
		
		// Compute the mean position of the quartet
		double meanX = 0.0;
		double meanY = 0.0;
		
		for(Triplet p: trajectories){
			meanX += p.getA();
			meanY += p.getB();
		}
		
		meanX = meanX/trajectories.length;
		meanY = meanY/trajectories.length;
		setMean(meanX, meanY);
		
		Triplet meanT = new Triplet (meanX, meanY, 0).toPolar (); // May be needed
		
		// Compute max distance to the mean position
		double maxDist = 0.0;

		for (int i = 0; i < trajectories.length; i++) { // For each trajectory
			Triplet p = trajectories[i];

			double dist =0.0;
			switch(distanceMetric){
				case EUCLIDEAN:
					dist = Math.sqrt ((p.getA () - meanX) * (p.getA () - meanX) + (p.getB () - meanY) * (p.getB () - meanY));
				break;
				case PATH:
					dist = Math.abs (p.getA () - meanT.getA ()) * (p.getB ());
				break;
				case ANGULAR:
					dist = Math.abs (p.getA () - meanT.getA ()) * (p.getB ());
				break;
			}
			
			if(dist > maxDist){
				maxDist = dist;
				maxPosition=i;
			}
		}
		
		variance = maxDist;
	}
}
