// This class is where the Nexus data structures will be built and tested...
// Note a file may have an incorrect data type and/or character alphabet
// We need to try to infer and display warnings....

package visrd;

import pal.io.*;

import java.io.*;
import java.util.*;

public abstract class NewNexusDocument{
	private static NexusTokenizer ntp;
	private static Taxon[] taxon;
	private static boolean mustDetectLabels=true; // If Taxa block missing or does not provide labels
	private static boolean isInterleaved=false; // Character data interleaved

	private static boolean dataTypeDetermined=false; // Have we yet determined the data type?
	private static int numberOfReplacedChars=0;
	
	public static void main(String[] args){
		System.out.println("Welcome to the nexus tester");
//		NewNexusDocument nnd = new NewNexusDocument();
//		nnd.readNexus(new FileReader("C:\\Program Files\\JCreatorV3\\MyProjects\\VisRD\\Nexus\\bees.nex"));
	}
	
	public static Taxon[] loadFile(File f){
		taxon=null;
		readNexus(f);
		
		return taxon;
	}
	
	public static void readNexus(File f){
		try{
			ntp = new NexusTokenizer(new PushbackReader(new FileReader(f)));
			ntp.setReadWhiteSpace(false); // ignore whitespace
			ntp.setIgnoreComments(true); // ignore comments
			String nToken = ntp.readToken(); // all tokens in uppercase
				
			while(nToken != null) {
				if(isValidToken(nToken)){
				//	System.out.println("Token: [" + nToken + "] Col: " + ntp.getCol() + " , Row: " + ntp.getRow());
					if(nToken.equalsIgnoreCase("begin")){
						parseBlock();
					}
				}
				//System.out.println();
				nToken = ntp.readToken();
			}
			
			if(numberOfReplacedChars>0){
				VisRDGUI.displayError(numberOfReplacedChars + " unrecognised characters replaced with gaps");
			}
		}
		catch(Exception e){
			e.printStackTrace();
		}
	}
	
	public static Taxon[] getTaxa(){
		return taxon;
	}
	
	public static boolean isValidToken(String nToken){
		if(ntp.getCol()==0){ return false; }
		if(nToken.equals("'")){ return false; }
		
		return true;
	}
	
	public static void parseBlock() throws Exception, IOException, NexusParseException{
		String blockName = ntp.readToken();
		System.out.println("Parsing block " + blockName);
		String nToken = ntp.readToken(); // all tokens in uppercase
		
		if(blockName.equalsIgnoreCase("taxa")){
			while(!nToken.equalsIgnoreCase("end")&&!nToken.equalsIgnoreCase("_detect_")){
				if(isValidToken(nToken)){
					if(nToken.equalsIgnoreCase("ntax")){
						nToken=ntp.readToken();nToken=ntp.readToken();
						taxon = new Taxon[Integer.parseInt(nToken)];
					}
					else if(nToken.equalsIgnoreCase("taxlabels")){
						mustDetectLabels=false;
						nToken = ntp.readToken();
						int i=0;
						
						while(!nToken.equalsIgnoreCase(";")){
							if(isValidToken(nToken)){ taxon[i++]=new Taxon(nToken); }
							nToken = ntp.readToken();
						}
					}
				}
				nToken = ntp.readToken();
			}
			
			if(nToken.equalsIgnoreCase("_detect_")){ mustDetectLabels=true; }
			
//			System.out.println(taxonToString(false));
		}
		else if(blockName.equalsIgnoreCase("characters")){
	//		System.out.println("Reading characters, must detect labels: " + mustDetectLabels);
			
			while(!nToken.equalsIgnoreCase("end")) {
				if(isValidToken(nToken)){
					if(nToken.equalsIgnoreCase("nchar")){
						nToken=ntp.readToken();nToken=ntp.readToken();
						
						for(int i=0;i<taxon.length;i++){
							taxon[i].initializeCharacterMatrix(Integer.parseInt(nToken));
						}
					}
					else if(nToken.equalsIgnoreCase("format")){ parseChacterFormat(); }
					else if(nToken.equalsIgnoreCase("matrix")){ parseChacterMatrix(); }
				}
				nToken = ntp.readToken();
			}
			
	//		System.out.println("\n\n");
//			
	//		System.out.println("DataType: " + Taxon.getDataTypeName());
	//		System.out.println("Symbols: " + Taxon.getDataTypeSymbols());
//			System.out.println(taxonToString(true));
		}
		else{
			System.out.println("WARNING: BLOCK '" + blockName.toUpperCase() + "' NOT RECOGNISED");
		}
	}
	
	private static void parseChacterFormat() throws Exception, IOException{
		String nToken= ntp.readToken();
		
		while(!nToken.equals(";")) {
			if(isValidToken(nToken)){
				if(nToken.equalsIgnoreCase("datatype")){
					nToken=ntp.readToken();nToken=ntp.readToken();
					
					// Check if the datatype is recognised
					try{
						Taxon.setDataType(nToken);
						dataTypeDetermined=true;
					}
					catch(Exception e){
						// Data type did not match known data types
						VisRDGUI.message("WARNING: DataType '" + nToken + "' not recognised!, attempting to infer datatype from alphabet");
					}
				}
				else if(nToken.equalsIgnoreCase("symbols")){
					ntp.readToken(); ntp.readToken(); nToken=ntp.readToken();
					
					StringBuffer symbols = new StringBuffer();
					while(!nToken.equalsIgnoreCase("\"")){
						if(!nToken.equalsIgnoreCase(" ")){
							symbols.append(nToken);
						}
						nToken=ntp.readToken();
					}
				
					if(!dataTypeDetermined){
						try{
							Taxon.setDataTypeFromSymbols(symbols.toString());
							dataTypeDetermined=true;
						}
						catch(Exception e){
							VisRDGUI.message("WARNING: The DataType in the Nexus file was not recognised and unable to infer from the symbols");
						}
					}
					else{ // Check these symbols are consistent with the datatype
						if(Taxon.symbolsMatchDataType(symbols.toString())){
							VisRDGUI.message("Symbols '" + symbols.toString() + "' consistent with " + Taxon.getDataTypeName() + " DataType");
						}
						else{
							VisRDGUI.message("WARNING: Symbols '" + symbols.toString() + "' inconsistent with " + Taxon.getDataTypeName() + " DataType, only recognised ambiguious characters will be resolved");
						}
					}
				}
				else if(nToken.equalsIgnoreCase("interleave")){ isInterleaved=true; }
			}
			nToken = ntp.readToken();
		}
	}
	
	private static void parseChacterMatrix() throws Exception, IOException{
		String symbols = Taxon.getDataTypeSymbols();
		
		String nToken = ntp.readToken();
		int taxonPointer=-1;
		
		int lastRow=-1;
		
		while(!nToken.equals(";")) {
			if(isValidToken(nToken)){
				if(nToken.equals("-")){
					taxon[taxonPointer].appendCharToMatrix('-');
			//		System.out.println("Generic code [" + nToken + "] for " + taxon[taxonPointer].getLabel());
				}
				else{
					// Does the line equal a taxon name?
					boolean isTaxonName=false;
					for(int i=0;i<taxon.length&&isTaxonName==false;i++){
						if(taxon[i].isLabelled(nToken)){
							taxonPointer=i;
							isTaxonName=true;
						}
					}
					
					if(!isTaxonName){
					//	System.out.println("{" + nToken + "} is NOT a label");
						appendDataToTaxa(symbols, nToken, taxonPointer);
					}
					else{
					//	System.out.println("{" + nToken + "} is a label");
					}
				}
			}
			nToken = ntp.readToken();
		}
	}
	
	public static void appendDataToTaxa(String symbols, String nToken, int taxonPointer){
		nToken = nToken.toLowerCase();
		
		for(int i=0;i<nToken.length();i++){
			taxon[taxonPointer].appendCharToMatrix(nToken.charAt(i));
		}
	}
	
	public static void taxaHaveChanged(Taxon[] newTaxon){
		taxon=newTaxon;
	}
	
	public static String taxonToString(boolean showCharacterData){
		StringBuffer sb = new StringBuffer("This is my set of taxon:\n\n");
		for(int i=0;i<taxon.length;i++){
			if(showCharacterData){ sb.append(taxon[i]); }
			else{ sb.append(taxon[i].getLabel()); }
			sb.append("\n\n");
		}
		return sb.toString();
	}
}