package visrd;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;

/*

    Created using IntelliJ IDEA.
    User: forslund
    Date: Dec 10, 2003
    Time: 12:15:53 AM
 
*/

public class HighwayPanel extends VisRDPanel implements MouseListener{
    private BufferedImage theImage;

	private int noQuartets, beginIndex, endIndex;
	private int graphHeight, graphWidth, diaWidth;

	private double deltaThreshold;

	private boolean useStepwise = false;

	private ArrayList<Integer> markers;
	private int misPos=0;
	
	public HighwayPanel (QuartetSet newData){
		super(newData, 940, 460);
		newData.multiplySupportByAlpha();
		
		addMouseListener(this);
		setPreferredSize (new Dimension (940, 460));
		
        setMinimumSize (new Dimension (940, 460));
        setMaximumSize (new Dimension (940, 460));

		markers = new ArrayList<Integer> ();

		graphWidth = 900;
		graphHeight = 400;

        theImage = new BufferedImage (940, 460, BufferedImage.TYPE_INT_ARGB);
	}
	
	public void mouseClicked(MouseEvent e){
		int sPos = (e.getX() - 40) * diaWidth / graphWidth + beginIndex;

		addMarkerGraph(sPos);
	}
	
	public void mouseEntered(MouseEvent e){}
	public void mouseExited(MouseEvent e){}
	public void mousePressed(MouseEvent e){
		// Movement Info sPos
		misPos = (e.getX() - 40) * diaWidth / graphWidth + beginIndex;
	}
	
	public void mouseReleased(MouseEvent e){
		int ePos = (e.getX() - 40) * diaWidth / graphWidth + beginIndex;
		int delta = Math.abs(ePos-misPos);
		
		if(delta>5){
			MovementInfo myWindow = new MovementInfo(misPos, ePos, beginIndex, endIndex);
		}
	}

    protected void paintComponent (Graphics g){
        g.drawImage (theImage, 0, 0, null);
    }

    public void redraw(){
		draw ((Graphics2D)theImage.getGraphics ());
		repaint();
    }

	protected void draw(Graphics2D g2, boolean useThread){
		final Graphics2D g=g2;
		g.setColor (Color.WHITE);
		g.fillRect (0, 0, 940, 460);

		diaWidth = (endIndex - beginIndex);

		//	Lane lines
		g.setColor (Color.BLUE);

		g.drawLine (40, graphHeight * 2 / 3, 40 + graphWidth, graphHeight * 2 / 3);
		g.drawLine (40, graphHeight / 3, 40 + graphWidth, graphHeight / 3);

		// Sequence scale
		g.setColor (Color.BLACK);

		for (int a = 0; a < 20; a++) {
			int pos = 40 + graphWidth * a / 20;
			int sPos = beginIndex + diaWidth * a / 20;


			g.drawLine (pos, 440, pos, 420);
			g.drawString ("" + sPos, pos+3, 430);
		}

		//Topology scale
		g.setColor (Color.BLACK);
		String[] marks = {"180", "140", "100", "60", "20", "-20", "-60", "-100", "-140", "-180"};

		for (int a = 0; a < 10; a++) {
			g.drawLine (0, a * graphHeight / 9, 10, a * graphHeight / 9);
			g.drawString (marks[a], 11, a * graphHeight / 9 + 10);
		}

		/**

			Then go through sufficient many quartets,
			and at each step... go through list until the one
			found is above startIndex, plot to proper position,
			plot to next using shortest path, proceed until below endIndex

		*/
		
		if(useThread){
			Thread drawThread = new Thread(){
				public void run(){ doDraw(g); }
			};
			drawThread.start();
		}
		else{ doDraw(g); }
	}
	
	protected void doDraw(Graphics2D g){
		for (int n = data.size () - 1; n > data.size () - 1 - noQuartets; n--) {
			if(n%50==0){
				try{ repaint(); }
				catch(Exception e){}
			}
			
			// For each quartet
	
			Triplet[] t = data.getTrajectory (n); // Get the trajectory list
			int maxPosition = data.getMaxPosition(n);
			
			Triplet previous = t[0];
			double previousPhi=previous.getPhi();
			double previousDistance=previous.getDistance();
			double currentPhi=0.0; double currentDistance=0.0;
	
			for (int p = 0; p < t.length; p++) {
				Triplet current = t[p];
	
				if (current.getC () > endIndex) {
					break; // Stop when the highest is reached...
				}
	
				if (current.getC () > beginIndex) {
					// Draw line previous -> current
					
					currentPhi = current.getPhi();
					currentDistance = current.getDistance();
	
					// How much does angle change?
					double deltaA = Math.min (Math.abs (currentPhi - previousPhi),
												2.0 - Math.abs (currentPhi - previousPhi));
	
	
					if (!useStepwise) {
						//Color determination using New scheme
						float aColor = (float) (1.0 - (previousDistance + currentDistance) / 2.0);
						
						if(p==maxPosition){
							g.setColor(Color.RED);
						}
						else if(quartetFlags[n]){
						//	g.setColor(Color.GREEN);
							g.setColor (new Color (0.0f, 1.0f, 0.0f , 1.0f - aColor));
						}
						else{
							g.setColor (new Color (aColor, aColor, aColor, 1.0f - aColor));
						}
					}
					else{
						// Color determination, Korbinians stepwise hack
						
						// Red, Green & Blue values are all the same as shade of grey
						if(quartetFlags[n]){ g.setColor(Color.GREEN); }
						else{
						    // init colors (black and opaque)
							int shade = 0;
							int alpha = 255;
	
							// mean distance to centre of triangle (B)
	
							double avgd = (previousDistance + currentDistance) / 2;
	
							// choose color depending on avgd
	
							if (avgd > 0.8) { // close to corner
								shade = 0; // black
							}
							else if (avgd > 0.4) {// closer to centre
								shade = 180;
							}
							else {
								shade = 240; // even greyer
							}
							g.setColor (new Color (shade, shade, shade, alpha));
						}
					}
	
	
					// Now, see if line is above the delta threshold...
					if (deltaA < deltaThreshold) {
						int x1 = ((int)previous.getC () - beginIndex) * graphWidth / diaWidth;
						int x2 = ((int)current.getC () - beginIndex) * graphWidth / diaWidth;
						int y1 = (int)((previousPhi + 1.0) * graphHeight / 2);
						int y2 = (int)((currentPhi + 1.0) * graphHeight / 2);
	
						if (Math.abs (currentPhi - previousPhi) < 1.0) {
							// Draw inside
	/*
							double x1 = (previous.getC () - beginIndex) * graphWidth / diaWidth;
							double x2 = (current.getC () - beginIndex) * graphWidth / diaWidth;
							double y1 = (previousPhi + 1.0) * graphHeight / 2;
							double y2 = (currentPhi + 1.0) * graphHeight / 2;
	*/
							g.drawLine (40 + x1, y1, 40 + x2, y2);
						}
						else {
							/**
								Draw outside... so....Up or down?
	
								If up, shortest path must be up. This requires previous to be in the upper half.
								So if previous is in upper, and distance between is more than one... the other
								must be below...
							*/
	
							if (previousPhi > 0) { // Go up...Intercept with axis is...
								
	/*
								double x1 = (previous.getC () - beginIndex) * graphWidth / diaWidth;
								double x2 = (current.getC () - beginIndex) * graphWidth / diaWidth;
								double y1 = (previousPhi + 1.0) * graphHeight / 2;
								double y2 = (currentPhi + 1.0) * graphHeight / 2;
	*/								int xm = (int)((double)(graphHeight - 1 - y1) / (double)(graphHeight - y1 + y2)) * (x2 - x1) + x1;
	
								g.drawLine (40 + x1, y1, 40 + xm, graphHeight - 1);
								g.drawLine (40 + xm, 0, 40 + x2, y2);
							}
							else { // Go down...
	/*
								double x1 = (previous.getC () - beginIndex) * graphWidth / diaWidth;
								double x2 = (current.getC () - beginIndex) * graphWidth / diaWidth;
								double y1 = (previousPhi + 1.0) * graphHeight / 2;
								double y2 = (currentPhi + 1.0) * graphHeight / 2;
	*/								int xm = (int)((double)y1 / (double)(graphHeight + y1 - y2)) * (x2 - x1) + x1;
	
								g.drawLine (40 + x1, y1, 40 + xm, 0);
								g.drawLine (40 + xm, graphHeight - 1, 40 + x2, y2);
							}
						}
					}
				}
	
				previous = current;
				previousPhi=currentPhi;
				previousDistance=currentDistance;
			}
		}
		
	
		// Marker lines
		g.setColor (Color.RED);
	
		for (int a = 0; a < markers.size (); a++) {
			int sPos = (markers.get (a)).intValue ();
			int xPos = 40 + (sPos - beginIndex) * graphWidth / diaWidth;
	
			g.drawLine (xPos, 0, xPos, graphHeight);
		}
	}

	public void set (int newNoQuartets, int newBeginIndex, int newEndIndex, double newDeltaThreshold,
					boolean newUseStepwise)
	{
		noQuartets = newNoQuartets;
		beginIndex = newBeginIndex;
		endIndex = newEndIndex;
		deltaThreshold = newDeltaThreshold;

		useStepwise = newUseStepwise;

		if (noQuartets > data.size ()) { noQuartets = data.size (); }

		if (noQuartets < 1) { noQuartets = 1; }
	}

	public void addMarkerGraph (int sPos){
		for(int i=0;i<markers.size();i++){
			int difference=markers.get(i).intValue()-sPos;
			if(difference<3&&difference>-3){
				markers.remove(i);
				redraw ();
				return;
			}
		}

		markers.add (new Integer (sPos));
		redraw ();
	}
	
	public void autoMarkers (){
		AutoMarkerPrefs amp = new AutoMarkerPrefs(this);
	}

	public void clearMarkers (){
		markers.clear ();
		redraw ();
	}
	
	private class AutoMarkerPrefs extends JFrame implements ActionListener{
		private HighwayPanel parent;
		private JButton start;
		private JSpinner deltaSpinner, percentageSpinner;
		
		public AutoMarkerPrefs(HighwayPanel parent){
			super("Auto marker preferences");
			this.parent = parent;
			
			JPanel mainPane = new JPanel(new GridBagLayout());
			GridBagConstraints c = new GridBagConstraints();
	        c.fill = GridBagConstraints.BOTH;
	        c.weightx = 0.0;
			c.gridx = 0;
			c.gridy = 0;
			
			mainPane.add(new JLabel("Movement threshold:"),c);
			c.gridx = 1;
			SpinnerNumberModel movementModel = new SpinnerNumberModel (0.30, 0, 1, 0.05);
			deltaSpinner = new JSpinner(movementModel);
			deltaSpinner.setPreferredSize(new Dimension(50,20));
			mainPane.add(deltaSpinner,c);

			c.gridx = 0;
			c.gridy = 1;
			mainPane.add(new JLabel("% Quartets threshold:"),c);
			c.gridx = 1;
			SpinnerNumberModel percentageModel = new SpinnerNumberModel (0.30, 0, 1, 0.05);
			percentageSpinner = new JSpinner(percentageModel);
			percentageSpinner.setPreferredSize(new Dimension(50,20));
			mainPane.add(percentageSpinner,c);

			c.gridx = 0;
			c.gridy = 2;
			c.gridwidth = 2;
			start = new JButton("Start");
			start.addActionListener(this);
			mainPane.add(start,c);

			getContentPane().add(mainPane);
			pack();
			show();	
		}
		
		
		public void actionPerformed(ActionEvent e){
			if(e.getSource()==start){
				int stepSize = VisRD.getStepSize ();
				int displayQuartets = VisRD.getDisplayQuartets();
				
				int size = data.getTrajectory(0).length;
				double delta = ((Double)deltaSpinner.getValue()).doubleValue();
			
				double percentage = ((Double)percentageSpinner.getValue()).doubleValue();			
				int quartetThreshold = (int)((double)displayQuartets*percentage);
				
				int newMarkers=0;
				
				for (int s = 0; s < size-1; s++) {
					// Compare the angle of each quartet at the start and end position
					int quartetCount=0;
					for (int n = data.size () - 1; n > data.size () - 1 - displayQuartets; n--) {
						Triplet[] tl = data.getTrajectory(n);
						
						Triplet t1 = tl[s].toPolar();
						Triplet t2 = tl[s+1].toPolar();
						
						double deltaA = Math.min (Math.abs (t2.getA () - t1.getA ()),
										2.0 - Math.abs (t2.getA () - t1.getA ()));
										
						if(deltaA>delta){ quartetCount++; }
					}
					
					if(quartetCount>quartetThreshold){
						newMarkers++;
						markers.add (new Integer (s*stepSize));
					}	
				}
				
				if(newMarkers==0){
					VisRDGUI.displayError("0 new markers determined, try setting lower thresholds.");
				}
				else{
					parent.redraw ();	
				}
			}
		}
		
	}
	


	private class MovementInfo extends JFrame implements ChangeListener{
		private static final int DEFAULT_WINDOW_SIZE=50;
		private Taxon[] taxon;
		private int[] taxaCountForAllQuartets;
		private int quartetsMoved;
		
		private JLabel quartetCount;
		private JList taxaResults;
		private JSpinner startSpinner, endSpinner, deltaSpinner;

		public MovementInfo(int sPos, int ePos, int minimumFrame, int maximumFrame){
			super("Movement info " + sPos + " to " + ePos);
	//		setSize(400,300);
			
			JPanel contentPane = new JPanel();
			contentPane.setLayout(new BorderLayout());
			
			JPanel topPane = new JPanel(new GridBagLayout());
			
			GridBagConstraints c = new GridBagConstraints();
			c.fill = GridBagConstraints.HORIZONTAL;
			c.ipady = 10;      //make this component tall

			c.weightx = 0.5;
			c.gridx = 0;
			c.gridy = 0;
			topPane.add(new JLabel("Start frame: "), c);

			startSpinner = new JSpinner(new SpinnerNumberModel (sPos, minimumFrame, maximumFrame, 1));
			startSpinner.addChangeListener(this);
			c.gridx = 1;
			c.gridy = 0;
			topPane.add(startSpinner, c);

			c.gridx = 2;
			c.gridy = 0;
			topPane.add(new JLabel("End frame: "), c);

			endSpinner = new JSpinner(new SpinnerNumberModel (ePos, minimumFrame, maximumFrame, 1));
			endSpinner.addChangeListener(this);
			c.gridx = 3;
			c.gridy = 0;
			topPane.add(endSpinner, c);
			
			c.gridwidth = 2;
			c.gridx = 0;
			c.gridy = 1;
			topPane.add(new JLabel("Movement threshold: "), c);

			deltaSpinner = new JSpinner(new SpinnerNumberModel (0.3, 0, 1, 0.05));			
			deltaSpinner.addChangeListener(this);
			c.gridx = 2;
			c.gridy = 1;
			topPane.add(deltaSpinner, c);

			quartetCount = new JLabel();
			c.weightx = 0.0;
			c.gridwidth = 4;
			c.gridx = 0;
			c.gridy = 2;
			topPane.add(quartetCount, c);

			contentPane.add(topPane, BorderLayout.NORTH);

			taxaResults = new JList();
			contentPane.add(new JScrollPane(taxaResults),BorderLayout.CENTER);
			
			displayTaxaResults();

			add(contentPane);
			pack();
			show();
		}
		
		public void stateChanged(ChangeEvent e){
			displayTaxaResults();
		}
		
		private void setTitle(int sPos, int ePos){
			setTitle("Movement info " + sPos + " to " + ePos);
		}
		
		private void displayTaxaResults(){
			Integer s = (Integer)startSpinner.getValue();
			int sPos = s.intValue();
						
			Integer e = (Integer)endSpinner.getValue();
			int ePos = e.intValue();
			
			Double d = (Double)deltaSpinner.getValue();
			double delta = d.doubleValue();

			setTitle(sPos, ePos);
			taxon=VisRD.getAllTaxon();

			
			String[] names=null; // Either the taxa or group names
			int[] taxaCount = getTaxaCount(sPos,ePos,delta);
			
			
			taxaCountForAllQuartets = new int[taxon.length];

				
			for(int i=0;i<data.size ();i++){
				Quartet q = data.getQuartet(i);
				
				taxaCountForAllQuartets[q.getX()]++;
				taxaCountForAllQuartets[q.getY()]++;
				taxaCountForAllQuartets[q.getU()]++;
				taxaCountForAllQuartets[q.getV()]++;
			}
			
			
			
			if(VisRD.getClusterQuartets()){
				Object[] groupNames = VisRD.getGroupNames();
				
				names = new String[groupNames.length];
				for(int i=0;i<groupNames.length;i++){
					names[i] = (String)groupNames[i];
				}
				
				int[][] taxaGroup = VisRD.getTaxaMatrix();
				int[] groupForTaxa = new int[taxon.length];
				for(int group=1;group<taxaGroup.length;group++){
					for(int i=0;i<taxaGroup[group].length;i++){
						groupForTaxa[taxaGroup[group][i]]=group-1;
					//	System.out.println("Taxa " + taxon[taxaGroup[group][i]].getLabel() + " is in group " + names[group-1] + " number=[" + (group-1) + "]");
					}
				}
				
				int[] taxaCountGRP = new int[taxaGroup.length-1];
				int[] taxaCountForAllQuartetsGRP = new int[taxaGroup.length-1];
				
				for(int i=0;i<taxaCount.length;i++){
					taxaCountGRP[groupForTaxa[i]]+=taxaCount[i];
				//	System.out.println("Taxa count for " + taxon[i].getLabel() + ": " + taxaCount[i] + " in group " + names[groupForTaxa[i]] );
					taxaCountForAllQuartetsGRP[groupForTaxa[i]]+=taxaCountForAllQuartets[i];
				}
				
				taxaCount=taxaCountGRP;
				taxaCountForAllQuartets=taxaCountForAllQuartetsGRP;
			}
			else{
				// Count up the number of times each taxa appears in quartets
				names = new String[taxon.length];
				
				for(int i=0;i<taxon.length;i++){
					names[i] = taxon[i].getLabel();
				}
			}
			
			
			
			
			
			
			quartetCount.setText(quartetsMoved + " Quartets moved, in which the following taxa are involved:-");


			TaxonResult[] result = new TaxonResult[names.length];
			
			for(int i=0;i<result.length;i++){
				result[i] = new TaxonResult(names[i],taxaCount[i],taxaCountForAllQuartets[i]);
			}
			
			java.util.Arrays.sort(result);
			taxaResults.setListData(result);
		}
		
		private int[] getTaxaCount(int sPos, int ePos, double deltaThreshold){
			// Returns the number of times each taxon appears in a quartet which breaks the threshold
			
			int[] taxaCount = new int[taxon.length];
			quartetsMoved=0;
			
		// Given 2 positions in the alignment we return the difference between theta
		// for all quartets, we therefore return q Triplets where q is num quartets

			for (int n = data.size () - 1; n > data.size () - 1 - VisRD.getDisplayQuartets(); n--) {
				Triplet[] t = data.getTrajectory (n); // Get the trajectory list

				// Position number uses the following equation:
				// ((Pos - beginIndex) / (endIndex - beginIndex)) * ((endIndex - beginIndex)/stepSize)
				// Simplifies to (Pos - beginIndex) / stepSize
				int sPosNumber = (sPos-beginIndex)/VisRD.getStepSize();
				int ePosNumber = (ePos-beginIndex)/VisRD.getStepSize();
				
				if(ePosNumber>=t.length){ ePosNumber=t.length-1; }

		//		System.out.println(sPosNumber + "," + ePosNumber);
				Triplet startTriplet = t[sPosNumber];
				Triplet endTriplet = t[ePosNumber];
				
				Triplet t1 = startTriplet.toPolar ();
				Triplet t2 = endTriplet.toPolar ();

				//How much does angle change?
				double deltaA = Math.min (Math.abs (t2.getA () - t1.getA ()),
										2.0 - Math.abs (t2.getA () - t1.getA ()));
				
				if(deltaA < deltaThreshold){
					quartetsMoved++;
					Quartet q = data.getQuartet(n);
					taxaCount[q.getX()]++;
					taxaCount[q.getY()]++;
					taxaCount[q.getU()]++;
					taxaCount[q.getV()]++;
				}
				
			//	System.out.println("DeltaA for quartet " + n + ": " + deltaA);
			}
			
			return taxaCount;
		}
		
		private class TaxonResult implements Comparable{
			private String name;
			private int numberOfQuartets;
			private double percentage;
			
			public TaxonResult(String name, int numberOfQuartets, int totalQuartets){
				this.name=name;
				this.numberOfQuartets=numberOfQuartets;
				percentage=100.0*((double)numberOfQuartets/(double)totalQuartets);
			}
			
			public int compareTo(Object rhs){
				TaxonResult otherResult = (TaxonResult)rhs;
			//	return otherResult.getNumberOfQuartets() - numberOfQuartets;
				return Double.compare(otherResult.getPercentage(),percentage);
			}
			
			public int getNumberOfQuartets(){ return numberOfQuartets; }
			public double getPercentage(){ return percentage; }
			
			public String toString(){
				// A nice String representation showing the % etc.
				String percentageString = Double.toString(percentage);
				if(percentageString.length()>5){ percentageString = percentageString.substring(0,5); }
				return name + " - " + numberOfQuartets + " (" + percentageString + "% of all quartets)";
			}
		}
	}
}
